package ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata;

import java.util.concurrent.TimeUnit;

import net.spy.memcached.MemcachedClient;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppData;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppId;
import ru.yandex.chemodan.app.dataapi.utils.ProtobufTranscoder;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Denis Bakharev
 */
public class MemcachedAppDataStorage implements AppDataStorage {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final ProtobufTranscoder<AppData> transcoder = new ProtobufTranscoder<>(AppData.class);
    private final Option<MemcachedClient> memcachedClientO;
    private final int expireSeconds;
    private final int getTimeoutMillis;

    public MemcachedAppDataStorage(
            Option<MemcachedClient> memcachedClientO,
            int expireSeconds,
            int getTimeoutMillis)
    {
        this.memcachedClientO = memcachedClientO;
        this.expireSeconds = expireSeconds;
        this.getTimeoutMillis = getTimeoutMillis;
    }

    @Override
    public Option<AppData> find(AppId appId) {
        Option<AppData> result = Option.empty();
        if(memcachedClientO.isPresent() && !isIgnoreReadFromMemcachedAppDataCache()) {
            try {
                AppData appData = memcachedClientO.get()
                        .asyncGet(appId.toString(), transcoder)
                        .get(getTimeoutMillis, TimeUnit.MILLISECONDS);
                result = Option.ofNullable(appData);
            } catch (Exception e) {
                logger.warn("Can't get from cache: {}", e);
            }
        }

        return result;
    }

    private boolean isIgnoreReadFromMemcachedAppDataCache() {
        boolean isIgnore = false;
        Option<MasterSlavePolicy> policy = MasterSlaveContextHolder.policyO();
        if(policy.isPresent() && (policy.get() == MasterSlavePolicy.RW_M || policy.get() == MasterSlavePolicy.R_M)) {
            isIgnore = true;
        }
        return isIgnore;
    }

    @Override
    public void set(AppData appData) {
        memcachedClientO.forEach(
                client -> client.set(appData.getId().toString(), expireSeconds, appData, transcoder)
        );
    }
}
