package ru.yandex.chemodan.app.dataapi.core.limiter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseAppContext;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContextSource;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppDatabaseSettings;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.commune.dynproperties.DynamicProperty;

/**
 * @author tolmalev
 */
public class DatabaseLimiter {

    public final DynamicProperty<ListF<String>> applicationWhitelist =
            new DynamicProperty<>("data-api.application.whitelist", Cf.arrayList());

    public final DynamicProperty<ListF<String>> applicationDatabasesCountWhitelist =
            new DynamicProperty<>("data-api.application.databases_count.whitelist", Cf.arrayList());

    private final DatabasesJdbcDao databasesDao;
    private final AppSettingsRegistry appSettingsRegistry;

    public DatabaseLimiter(DatabasesJdbcDao databasesDao, AppSettingsRegistry appSettingsRegistry) {
        this.databasesDao = databasesDao;
        this.appSettingsRegistry = appSettingsRegistry;
    }

    public void checkLimitReached(Database database) {
        if (isInWhitelist(database)) {
            return;
        }

        AppDatabaseSettings settings = appSettingsRegistry.getDatabaseSettings(database);

        if (database.meta.size.ge(settings.getDatabaseSizeLimit())) {
            throw new DatabaseSizeLimitException();
        }
        if (database.meta.recordsCount > settings.getDatabaseRecordsCountLimit().getOrElse(Integer.MAX_VALUE)) {
            throw new DatabaseSizeLimitException();
        }
    }

    public void checkDatabasesCountLimit(DataApiUserId uid, DatabaseContextSource ctxSrc) {
        if (isInWhitelist(ctxSrc) || isInDatabasesCountWhitelist(ctxSrc)) {
            return;
        }

        if (getDatabaseCount(uid, ctxSrc) >= appSettingsRegistry.getDefaultDatabasesCountLimit()) {
            throw new DatabasesCountLimitException();
        }
    }

    private boolean isInDatabasesCountWhitelist(DatabaseContextSource ctxSrc) {
        return ctxSrc.appContextO()
                .isMatch(this::isInDatabasesCountWhitelist);
    }

    public boolean isInWhitelist(DatabaseContextSource ctxSrc) {
        return ctxSrc.appContextO()
                .isMatch(this::isInWhitelist);
    }

    private boolean isInWhitelist(DatabaseAppContext context) {
        return applicationWhitelist.get()
                .containsTs(context.appName());
    }

    private boolean isInDatabasesCountWhitelist(DatabaseAppContext context) {
        return applicationDatabasesCountWhitelist.get()
                .containsTs(context.appName());
    }

    private int getDatabaseCount(DataApiUserId uid, DatabaseContextSource ctxSrc) {
        return databasesDao.findDatabasesCount(uid, ctxSrc.dbContext());
    }
}
