package ru.yandex.chemodan.app.dataapi.core.limiter;

import org.junit.Test;
import org.mockito.Matchers;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseAppContext;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefSource;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppDatabaseSettings;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.test.UnitTestBase;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.test.Assert;

import static org.mockito.Mockito.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

/**
 * @author Denis Bakharev
 */
public class DatabaseLimiterUnitTest extends UnitTestBase {
    private final DatabasesJdbcDao daoMock = mock(DatabasesJdbcDao.class);
    private final AppSettingsRegistry settingsRegistry = mock(AppSettingsRegistry.class);

    @Test
    public void simpleCheckTests() {
        mockFindDatabasesCount(100);
        mockSettingsRegistry();

        DatabaseLimiter dl = new DatabaseLimiter(daoMock, settingsRegistry);
        String appInWhitelist = "appInWhitelist";
        dl.applicationWhitelist.get().add(appInWhitelist);

        dl.checkLimitReached(getDatabase(appInWhitelist, 1000000000));
        dl.checkDatabasesCountLimit(getUserId(), new DatabaseAppContext(appInWhitelist));

        Assert.assertThrows(() -> dl.checkLimitReached(getDatabase("simpleApp", 1000000000)),
                            DatabaseSizeLimitException.class);

        Assert.assertThrows(() -> dl.checkDatabasesCountLimit(getUserId(), new DatabaseAppContext("simpleApp")),
                            DatabasesCountLimitException.class);
    }

    @Test
    public void differentSizeLimits() {
        AppDatabaseRef ref1 = new AppDatabaseRef("app1", "db1");

        AppDatabaseSettings settings = new AppDatabaseSettings(ref1)
                .setDatabaseSizeLimit(Option.of(DataSize.fromBytes(10)))
                .setDatabaseRecordsCountLimit(Option.of(100));

        when(settingsRegistry.getDatabaseSettings(matcher(ref1))).thenReturn(settings);


        AppDatabaseRef ref2 = new AppDatabaseRef("app1", "db2");

        settings = new AppDatabaseSettings(ref2)
                .setDatabaseSizeLimit(Option.of(DataSize.fromBytes(20)))
                .setDatabaseRecordsCountLimit(Option.of(200));

        when(settingsRegistry.getDatabaseSettings(matcher(ref2))).thenReturn(settings);


        AppDatabaseRef ref3 = new AppDatabaseRef("app2", "db1");

        settings = new AppDatabaseSettings(ref3)
                .setDatabaseSizeLimit(Option.of(DataSize.fromBytes(30)))
                .setDatabaseRecordsCountLimit(Option.of(300));

        when(settingsRegistry.getDatabaseSettings(matcher(ref3))).thenReturn(settings);

        DatabaseLimiter dl = new DatabaseLimiter(daoMock, settingsRegistry);



        dl.checkLimitReached(getDatabase(ref1, 9));
        Assert.assertThrows(() ->
                dl.checkLimitReached(getDatabase(ref1, 10)),
                DatabaseSizeLimitException.class);
        Assert.assertThrows(() ->
                        dl.checkLimitReached(getDatabase(ref1, 11)),
                DatabaseSizeLimitException.class);

        dl.checkLimitReached(getDatabase(ref2, 9));
        dl.checkLimitReached(getDatabase(ref2, 10));
        dl.checkLimitReached(getDatabase(ref2, 11));
        dl.checkLimitReached(getDatabase(ref2, 19));

        Assert.assertThrows(() ->
                        dl.checkLimitReached(getDatabase(ref2, 20)),
                DatabaseSizeLimitException.class);
        Assert.assertThrows(() ->
                        dl.checkLimitReached(getDatabase(ref1, 21)),
                DatabaseSizeLimitException.class);


        dl.checkLimitReached(getDatabase(ref3, 19));
        dl.checkLimitReached(getDatabase(ref3, 20));
        dl.checkLimitReached(getDatabase(ref3, 21));
        dl.checkLimitReached(getDatabase(ref3, 29));

        Assert.assertThrows(() ->
                        dl.checkLimitReached(getDatabase(ref2, 30)),
                DatabaseSizeLimitException.class);
        Assert.assertThrows(() ->
                        dl.checkLimitReached(getDatabase(ref1, 31)),
                DatabaseSizeLimitException.class);
    }

    private DatabaseRefSource matcher(final AppDatabaseRef ref) {
        return Matchers.argThat(refSource -> {
                if (!(refSource instanceof DatabaseRefSource)) {
                    return false;
                }
                return refSource.appNameO().equals(ref.appNameO())
                    && refSource.databaseId().equals(ref.databaseId());
            });
    }

    private void mockSettingsRegistry() {
        when(settingsRegistry.getDefaultDatabasesCountLimit()).thenReturn(20);

        AppDatabaseSettings settings = new AppDatabaseSettings(new AppDatabaseRef("", ""))
                .setDatabaseSizeLimit(Option.of(DataSize.fromBytes(10)));

        when(settingsRegistry.getDatabaseSettings(any())).thenReturn(settings);
    }

    private void mockFindDatabasesCount(int result) {
        when(daoMock.findDatabasesCount(any(), any())).thenReturn(result);
    }
}
