package ru.yandex.chemodan.app.dataapi.core.limiter.access;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Vec4;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseAliasType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.Monica;
import ru.yandex.misc.monica.MonicaConfiguration;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Gauge;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.FullMetricName;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.util.measure.Measured;
import ru.yandex.misc.monica.util.measure.Measurer;
import ru.yandex.misc.worker.AlarmThread;

/**
 * @author tolmalev
 */
public class DataApiManagerLimitingInvocationHandler implements InvocationHandler, MonicaContainer, PingerChecker {
    @SuppressWarnings("unused")
    private static final Logger logger = LoggerFactory.getLogger(DataApiManagerLimitingInvocationHandler.class);

    static final DynamicProperty<Boolean> enableCounting =
            new DynamicProperty<>("data-api-manager-counting-enabled",
                    EnvironmentType.getActive() != EnvironmentType.PRODUCTION);

    // our desktop application
    static final DynamicProperty<ListF<String>> whitelistApps =
            new DynamicProperty<>("dataapi-stats-measure-apps", Cf.list());

    private final DataApiManager target;
    private final DataApiAccessRateLimiter acessLimiter;

    private final ExternalDatabasesRegistry externalDatabasesRegistry;
    private final Option<TypeSettingsRegistry> typeSettingsRegistry;

    private volatile SetF<String> measuredHostApps = Cf.set();
    private final UpdateListThread workerThread;

    private CountDownLatch initializedLatch = new CountDownLatch(1);

    @MonicaMetric
    @GroupByDefault
    private final InstrumentMap byHostApp = new InstrumentMap();

    @MonicaMetric
    @GroupByDefault
    private final InstrumentMap byClient = new InstrumentMap();

    private final MapF<MetricName, AtomicInteger> countersByHostApp = Cf.concurrentHashMap();
    private final MapF<MetricName, AtomicInteger> countersByClient = Cf.concurrentHashMap();

    private final AtomicInteger registeredInMonica = new AtomicInteger();

    public DataApiManagerLimitingInvocationHandler(DataApiManager target,
            DataApiAccessRateLimiter acessLimiter,
            ExternalDatabasesRegistry externalDatabasesRegistry,
            //TODO: use interface instead of option
            Option<TypeSettingsRegistry> typeSettingsRegistry)
    {
        this.target = target;
        this.acessLimiter = acessLimiter;
        this.externalDatabasesRegistry = externalDatabasesRegistry;
        this.typeSettingsRegistry = typeSettingsRegistry;

        this.workerThread = new UpdateListThread();
        this.workerThread.startGracefully();
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        if (!enableCounting.get()) {
            return invokeSimple(method, args);
        }

        Option<UserDatabaseSpec> specO = extractSpec(args);
        if (!specO.isPresent()) {
            //nothing to measure
            return invokeSimple(method, args);
        }

        UserDatabaseSpec spec = specO.get();

        String databaseId = spec.databaseRef().databaseId();
        Option<String> hostAppO = spec.databaseRef().appNameO();

        Option<String> clientAppO = Option
                .when(spec.databaseAlias().aliasType() == DatabaseAliasType.EXTERNAL,
                        () -> ((ExternalDatabaseAlias) spec.databaseAlias()).clientAppName()
                ).orElse(hostAppO);

        if (!needMeasure(hostAppO)) {
            // don't need measure
            return invokeSimple(method, args);
        }

        MetricName byHostMetricName = new MetricName(hostAppO.getOrElse("global"), databaseId);
        MetricName byClientMetricName = new MetricName(clientAppO.getOrElse("global"),
                spec.databaseAlias().aliasType() == DatabaseAliasType.DIRECT
                        ? databaseId
                        : spec.toString()
        );

        Vec4<AtomicInteger> counters = Cf.vec(
                getOrCreate(countersByClient, new MetricName(byClientMetricName.asList().first())),
                getOrCreate(countersByClient, byClientMetricName),
                getOrCreate(countersByHostApp, new MetricName(byHostMetricName.asList().first())),
                getOrCreate(countersByHostApp, byHostMetricName));

        LimitingCountersValue limitingCounters = new LimitingCountersValue(
                counters.get(0).incrementAndGet(), counters.get(1).incrementAndGet(),
                counters.get(2).incrementAndGet(), counters.get(3).incrementAndGet());

        try {
            //may throw exception
            acessLimiter.checkLimits(clientAppO, hostAppO, databaseId, limitingCounters);

            return invokeInternal(method, args, byHostMetricName, byClientMetricName);

        } finally {
            counters.forEach(AtomicInteger::decrementAndGet);
        }
    }

    private boolean needMeasure(Option<String> hostAppO) {
        return measuredHostApps.containsTs(hostAppO.getOrElse(""));
    }

    private Object invokeSimple(Method method, Object[] args) throws Throwable {
        try {
            return method.invoke(target, args);
        } catch (IllegalAccessException e) {
            // impossible
            throw ExceptionUtils.translate(e);
        } catch (InvocationTargetException e) {
            throw e.getTargetException();
        }
    }

    private Object invokeInternal(Method method, Object[] args, MetricName byHostMetricName, MetricName byClientMetricName) {
        Measured measure = Measurer.I.measure(() -> {
            try {
                return method.invoke(target, args);
            } catch (IllegalAccessException e) {
                // impossible
                throw ExceptionUtils.translate(e);
            } catch (InvocationTargetException e) {
                if (e.getTargetException() instanceof Exception) {
                    throw ExceptionUtils.translate((Exception) e.getTargetException());
                } else {
                    throw ExceptionUtils.translate(e);
                }
            }
        });

        byHostApp.update(measure.info(), byHostMetricName, UpdateMode.RECURSIVE);
        byClient.update(measure.info(), byClientMetricName, UpdateMode.RECURSIVE);

        return measure.cont();
    }

    //XXX: it's not very important so I don't want to do it through constructor
    @Autowired(required = false)
    private Monica monica;
    @Autowired(required = false)
    private MonicaConfiguration monicaConfiguration;

    private AtomicInteger getOrCreate(MapF<MetricName, AtomicInteger> counters, MetricName name) {
        return counters.computeIfAbsent(name, key -> {
            AtomicInteger value = new AtomicInteger();

            if (monica != null && monicaConfiguration != null) {
                registerInMonica(counters, key, value);
            }
            return value;
        });
    }

    private void registerInMonica(MapF<MetricName, AtomicInteger> counters, MetricName name, final AtomicInteger value) {
        if (registeredInMonica.incrementAndGet() > 300) {
            // don't register too much in monica
            return;
        }

        String mapName = counters == countersByClient
                ? "countersByClient"
                : "countersByHostApp";

        MetricName withPrefix = groupName("").namePrefix
                .withSuffix(mapName)
                .withSuffix(name.asList());

        monica.registerMetric(
                FullMetricName.Factory.consRaw(monicaConfiguration.localNamespace(), withPrefix),
                Gauge.cons(value::get, Integer.class));
    }

    private Option<UserDatabaseSpec> extractSpec(Object[] args) {
        DataApiUserId uid = null;
        DatabaseRef databaseRef = null;
        for (Object arg : args) {
            if (arg instanceof UserDatabaseSpec) {
                return Option.of((UserDatabaseSpec) arg);
            }
            if (arg instanceof Database) {
                return Option.of(UserDatabaseSpec.fromDatabase((Database) arg));
            }
            if (arg instanceof DatabaseRefSource) {
                databaseRef = ((DatabaseRefSource) arg).dbRef();
            }
            if (arg instanceof DataApiUserId) {
                uid = (DataApiUserId) arg;
            }
        }
        if (uid != null && databaseRef != null) {
            return Option.of(new UserDatabaseSpec(uid, databaseRef));
        }
        return Option.empty();
    }

    public void waitInitialized() {
        try {
            initializedLatch.await();
        } catch (InterruptedException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Override
    public boolean isActive() {
        return initializedLatch.getCount() <= 0;
    }

    private class UpdateListThread extends AlarmThread {

        public UpdateListThread() {
            super("update-measured-apps", 500);
        }

        @Override
        protected void alarm() {
            SetF<String> newSet = Cf.hashSet();
            newSet.addAll(externalDatabasesRegistry.getAll().map(p -> p.originalApp).unique());
            newSet.addAll(typeSettingsRegistry.flatMap(tsr -> tsr.getAllTypeSettings().flatMap(ts -> ts.dbRef().appNameO()).unique()));
            newSet.addAll(whitelistApps.get());

            measuredHostApps = newSet;

            if (initializedLatch.getCount() > 0) {
                initializedLatch.countDown();
            }
        }
    }

    @Override
    public MetricGroupName groupName(String s) {
        return new MetricGroupName(
                "data-api-manager",
                new MetricName("data-api", "manager"),
                "All called methods in DataApiManager"
        );
    }

    @MonicaMetric
    @GroupByDefault
    public int countersByClientSize() {
        return countersByClient.size();
    }

    @MonicaMetric
    @GroupByDefault
    public int countersByHostAppSize() {
        return countersByHostApp.size();
    }
}
