package ru.yandex.chemodan.app.dataapi.core.limiter.access;

import java.lang.reflect.Field;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.DatabaseChangedEventHandler;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.PatchableSnapshot;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseAccessType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author tolmalev
 */
@ContextConfiguration(classes = {
        DataApiManagerLimitingInvocationHandlerTest.Context.class
})
@ActivateDataApiEmbeddedPg
public class DataApiManagerLimitingInvocationHandlerTest extends DataApiTestSupport {
    @Autowired
    private AccessRateLimitsRegistry accessRateLimitsRegistry;

    @Autowired
    private ExternalDatabasesRegistry externalDatabasesRegistry;

    @Autowired
    private DataApiManagerLimitingInvocationHandler handler;

    @Autowired
    private DataApiManager dataApiManager;

    private DataApiUserId uid;

    @Before
    public void Before() {
        uid = createRandomCleanUserInDefaultShard();
        handler.waitInitialized();
    }

    @Test
    public void limitReached() {
        UserDatabaseSpec db1 = new UserDatabaseSpec(uid, DatabaseRef.cons(Option.of("app"), "db1"));
        UserDatabaseSpec db2 = new UserDatabaseSpec(uid, DatabaseRef.cons(Option.of("app"), "db2"));
        UserDatabaseSpec db3 = new UserDatabaseSpec(uid, DatabaseRef.cons(Option.of("app"), "db3"));

        UserDatabaseSpec db1_ext = UserDatabaseSpec.fromUserAndAlias(uid, new ExternalDatabaseAlias("c_app", "app", "db1"));
        UserDatabaseSpec db2_ext = UserDatabaseSpec.fromUserAndAlias(uid, new ExternalDatabaseAlias("c_app", "app", "db2"));

        dataApiManager.getOrCreateDatabase(db1);
        dataApiManager.getOrCreateDatabase(db2);
        dataApiManager.getOrCreateDatabase(db3);

        ForkJoinPool pool = new ForkJoinPool(100);

        // applyDelta will wait for 5 seconds, so we have time to check
        Cf.range(0, 5).forEach(i ->
                pool.submit(() -> dataApiManager.applyDelta(db1, 0, RevisionCheckMode.PER_RECORD, Delta.empty())));

        Cf.range(0, 5).forEach(i ->
                pool.submit(() -> dataApiManager.applyDelta(db1_ext, 0, RevisionCheckMode.PER_RECORD, Delta.empty())));

        Cf.range(0, 10).forEach(i ->
                pool.submit(() -> dataApiManager.applyDelta(db2, 0, RevisionCheckMode.PER_RECORD, Delta.empty())));

        //hack to be sure that counters has been incremented
        ThreadUtils.sleep(100);

        // limit for db1 not reached for host app
        dataApiManager.getDatabase(db1);

        // limit for db1 is reached for client app
        Assert.assertThrows(
                () -> dataApiManager.getDatabase(db1_ext),
                AccessRateLimitException.class
        );

        // limit for db2 is reached for host app (so reached for client too)
        Assert.assertThrows(
                () -> dataApiManager.getDatabase(db2),
                AccessRateLimitException.class
        );
        Assert.assertThrows(
                () -> dataApiManager.getDatabase(db2_ext),
                AccessRateLimitException.class
        );
    }

    @Test(expected = OutdatedChangeException.class)
    public void rightException() {
        UserDatabaseSpec db1 = new UserDatabaseSpec(uid, DatabaseRef.cons(Option.of("app"), "db1"));

        dataApiManager.getOrCreateDatabase(db1);
        dataApiManager.applyDelta(db1, 0, RevisionCheckMode.WHOLE_DATABASE, Delta.empty());
        dataApiManager.applyDelta(db1, 0, RevisionCheckMode.WHOLE_DATABASE, Delta.empty());
    }

    @Test(expected = OutdatedChangeException.class)
    public void rightExceptionWithCountingDisabled() throws NoSuchFieldException, IllegalAccessException {
        UserDatabaseSpec db1 = new UserDatabaseSpec(uid, DatabaseRef.cons(Option.of("app"), "db1"));

        DynamicProperty<Boolean> dp = DataApiManagerLimitingInvocationHandler.enableCounting;
        Field field = dp.getClass().getDeclaredField("value");
        field.setAccessible(true);

        ((AtomicReference) field.get(dp)).set(false);

        try {
            dataApiManager.getOrCreateDatabase(db1);
            dataApiManager.applyDelta(db1, 0, RevisionCheckMode.WHOLE_DATABASE, Delta.empty());
            dataApiManager.applyDelta(db1, 0, RevisionCheckMode.WHOLE_DATABASE, Delta.empty());
        } finally {
            ((AtomicReference) field.get(dp)).set(true);
        }
    }

    static final class Context {
        @Bean
        public DatabaseChangedEventHandler waitingHandler() {
            return new DatabaseChangedEventHandler() {
                @Override
                public ListF<RecordChange> databaseChanged(PatchableSnapshot snapshot) {
                    ThreadUtils.sleep(5000);
                    return Cf.list();
                }

                @Override
                public int getOrder() {
                    return HIGHEST_PRECEDENCE;
                }
            };
        }

        @Bean
        @Primary
        public AccessRateLimitsRegistry accessRateLimitsRegistry() {
            AccessRateLimitsRegistry registry = Mockito.mock(AccessRateLimitsRegistry.class);

            MapF<String, AccessRateLimits> limits = Cf.list(
                    new AccessRateLimits(AccessRateLimits.Type.BY_HOST, "app", Option.empty(), 30),
                    new AccessRateLimits(AccessRateLimits.Type.BY_HOST, "app", Option.of("db1"), 15),
                    new AccessRateLimits(AccessRateLimits.Type.BY_HOST, "app", Option.of("db2"), 10),

                    new AccessRateLimits(AccessRateLimits.Type.BY_CLIENT, "c_app", Option.empty(), 20),
                    new AccessRateLimits(AccessRateLimits.Type.BY_CLIENT, "c_app", Option.of(".ext.app@db1"), 5)
            ).toMapMappingToKey(AccessRateLimitsRegistry::id);

            Mockito
                    .when(registry.getLimits(Mockito.any(), Mockito.any(), Mockito.any()))
                    .then(invocation -> {
                        Option<String> clientAppO = (Option<String>) invocation.getArguments()[0];
                        Option<String> hostAppO = (Option<String>) invocation.getArguments()[1];
                        String databaseId = (String) invocation.getArguments()[2];

                        String clientDbId = !clientAppO.isPresent() || !hostAppO.isPresent() || clientAppO.equals(hostAppO)
                                    ? databaseId
                                    : new ExternalDatabaseAlias(clientAppO.get(), hostAppO.get(), databaseId).toString();

                        return
                                clientAppO.flatMap(clientApp ->
                                        limits.getO(AccessRateLimitsRegistry.id(AccessRateLimits.Type.BY_CLIENT, clientApp))
                                                .plus(limits.getO(AccessRateLimitsRegistry.id(AccessRateLimits.Type.BY_CLIENT, clientApp, Option.of(clientDbId))))
                                ).plus(
                                        hostAppO.flatMap(hostApp ->
                                                limits.getO(AccessRateLimitsRegistry.id(AccessRateLimits.Type.BY_HOST, hostApp))
                                                        .plus(limits.getO(AccessRateLimitsRegistry.id(AccessRateLimits.Type.BY_HOST, hostApp, Option.of(databaseId))))
                                        )
                                );
                    });

            return registry;
        }

        @Bean
        @Primary
        public ExternalDatabasesRegistry externalDatabasesRegistry() {
            ExternalDatabasesRegistry registry = Mockito.mock(ExternalDatabasesRegistry.class);
            Mockito.when(registry
                    .getExternalDatabaseAccessType(Mockito.any()))
                    .thenReturn(Option.of(DatabaseAccessType.READ_WRITE)
            );

            Mockito.when(registry
                    .getAll())
                    .thenReturn(Cf.list(
                            new ExternalDatabasesRegistry.ExternalDatabasePojo("fake_c_app", "app", "db1", DatabaseAccessType.READ_WRITE, Option.empty()),
                            new ExternalDatabasesRegistry.ExternalDatabasePojo("fake_c_app", "app", "db2", DatabaseAccessType.READ_WRITE, Option.empty()),
                            new ExternalDatabasesRegistry.ExternalDatabasePojo("fake_c_app", "app", "db3", DatabaseAccessType.READ_WRITE, Option.empty())
                    ));

            return registry;
        }
    }
}
