package ru.yandex.chemodan.app.dataapi.core.logbroker;

import javax.annotation.Nullable;

import org.joda.time.Duration;

import ru.yandex.chemodan.app.dataapi.api.data.record.RecordRef;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dump.MethodType;
import ru.yandex.chemodan.app.dataapi.core.dump.RecordChangeEvent;
import ru.yandex.chemodan.app.dataapi.worker.DataApiTaskQueueName;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.digest.Md5;

/**
 * @author dbrylev
 */
public class LogBrokerPushRecordChangeTask extends OnetimeTaskSupport<LogBrokerPushRecordChangeTask.Parameters> {

    private LogBrokerPushSender sender;

    public LogBrokerPushRecordChangeTask(RecordChangeEvent event) {
        super(new Parameters(event));
    }

    public LogBrokerPushRecordChangeTask(LogBrokerPushSender sender) {
        super(Parameters.class);
        this.sender = sender;
    }

    @Override
    protected void execute(Parameters params, ExecutionContext context) throws Exception {
        sender.sendOrThrow(params.event);
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new CompoundReschedulePolicy(
                new RescheduleConstant(Duration.standardMinutes(1), 10),
                new RescheduleExponential(Duration.standardMinutes(1), 10));
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return ActiveUidConverter.class;
    }

    @Override
    public TaskQueueName queueName() {
        return DataApiTaskQueueName.DATAAPI_REGULAR;
    }

    @BenderBindAllFields
    public static class Parameters {
        @BenderFlatten
        public final RecordChangeEvent event;

        public Parameters(RecordChangeEvent event) {
            this.event = event;
        }
    }

    @BenderBindAllFields
    public static class ActiveUid {
        public final DataApiUserId uid;
        public final String recordId;

        public final MethodType method;
        public final String dataHash;

        public ActiveUid(DataApiUserId uid, String recordId, MethodType method, String dataHash) {
            this.uid = uid;
            this.recordId = recordId;
            this.method = method;
            this.dataHash = dataHash;
        }
    }

    public static class ActiveUidConverter implements ActiveUniqueIdentifierConverter<Parameters, ActiveUid> {
        @Override
        public Class<ActiveUid> getActiveUniqueIdentifierClass() {
            return ActiveUid.class;
        }

        @Override
        public ActiveUid convert(Parameters params) {
            RecordRef recRef = params.event.getRecordRef();
            long rev = params.event.recordRev;

            String recordId = recRef.dbRef() + "/" + recRef.idStr() + ":" + rev;
            String dataHash = Md5.A.digest(params.event.recordData).hex();

            return new ActiveUid(params.event.uid, recordId, params.event.method, dataHash);
        }
    }
}
