package ru.yandex.chemodan.app.dataapi.core.logbroker;

import java.io.Closeable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dump.RecordChangeEvent;
import ru.yandex.chemodan.logbroker.PushClientFactory;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.inside.logbroker.push.PushClient;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.util.measure.MeasureInfo;

/**
 * @author dbrylev
 */
public class LogBrokerPushSender implements Closeable, MonicaContainer {

    private final PushClientFactory pushClientFactory;
    private final MapF<String, PushClient> pushClients = Cf.concurrentHashMap();

    private final ReschedulePolicy reschedulePolicy;
    private final Duration timeout;

    private final AppSettingsRegistry appSettingsRegistry;

    @MonicaMetric
    @GroupByDefault
    private final MeterMap sentBytes = new MeterMap();
    @MonicaMetric
    @GroupByDefault
    private final InstrumentMap sendings = new InstrumentMap();

    public LogBrokerPushSender(
            PushClientFactory pushClientFactory,
            ReschedulePolicy reschedulePolicy, Duration timeout,
            AppSettingsRegistry appSettingsRegistry)
    {
        this.pushClientFactory = pushClientFactory;
        this.reschedulePolicy = reschedulePolicy;
        this.timeout = timeout;
        this.appSettingsRegistry = appSettingsRegistry;
    }

    public void sendOrThrow(RecordChangeEvent event) {
        Option<Exception> result = send(Cf.list(event)).single();

        if (result.isPresent()) {
            throw ExceptionUtils.translate(result.get());
        }
    }

    public ListF<Option<Exception>> send(ListF<RecordChangeEvent> events) {
        return events.map(this::supply).map(s -> {
            try {
                if (s.isPresent()) {
                    s.get().future.get(timeout.getMillis(), TimeUnit.MILLISECONDS);

                    sendings.update(s.get().measured(true), s.get().metricName(), UpdateMode.RECURSIVE);
                    sentBytes.inc(s.get().size, s.get().metricName(), UpdateMode.RECURSIVE);
                }
                return Option.empty();

            } catch (InterruptedException | ExecutionException | TimeoutException e) {
                sendings.update(s.get().measured(false), s.get().metricName(), UpdateMode.RECURSIVE);
                return Option.of(e);
            }
        });
    }

    private Option<Sending> supply(RecordChangeEvent event) {
        Option<String> typeName = appSettingsRegistry.getDatabaseSettings(event.getDbRef()).getLbPushLogTypeName();

        Option<String> logType = typeName.map(name -> "ydisk-dataapi-" + name + "-changes-log");

        if (logType.isPresent()) {
            PushClient client = pushClients.computeIfAbsent(logType.get(), pushClientFactory::buildAndStart);

            byte[] bytes = event.formatTskvLine().getBytes();
            Instant start = Instant.now();

            Future<?> future = client.sendData(bytes, reschedulePolicy);

            return Option.of(new Sending(future, typeName.get(), bytes.length, start));

        } else {
            return Option.empty();
        }
    }

    @Override
    public void close() {
        pushClients.values().forEach(PushClient::stop);
    }

    @Override
    public MetricGroupName groupName(String s) {
        return new MetricGroupName(
                "dataapi",
                new MetricName("dataapi", "lb-push-sender"),
                "Log broker changes pusher");
    }

    private static class Sending {
        private final Future<?> future;
        private final String typeName;
        private final int size;

        private final Instant started;

        public Sending(Future<?> future, String typeName, int size, Instant started) {
            this.future = future;
            this.typeName = typeName;
            this.size = size;
            this.started = started;
        }

        public MetricName metricName() {
            return new MetricName(typeName);
        }

        public MeasureInfo measured(boolean successful) {
            return new MeasureInfo(new Duration(started, Instant.now()), successful);
        }
    }
}
