package ru.yandex.chemodan.app.dataapi.core.manager;

import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;
import java.util.function.Supplier;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.dataapi.api.DatabaseChangedEventAsyncHandler;
import ru.yandex.chemodan.app.dataapi.api.DatabaseChangedEventHandler;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.PatchableSnapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotPojoRow;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotWithSource;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSource;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSession;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSessionDecorator;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceTrait;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.datasource.DsSessionTxManager;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseAccessType;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.StuckBehindDatabaseException;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseAliasType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaUtils;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaValidationMode;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasAppliedDatabase;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.deltas.TooNewChangeException;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.CompositeApplicationManager;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.DataApiStats;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabaseRevisionMismatchException;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.LockedDatabaseSession;
import ru.yandex.chemodan.app.dataapi.core.limiter.DatabaseLimiter;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceManager;
import ru.yandex.chemodan.app.dataapi.utils.PriorityRunnable;
import ru.yandex.chemodan.app.dataapi.web.AccessForbiddenException;
import ru.yandex.chemodan.app.dataapi.web.DatabaseNotFoundException;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.chemodan.app.dataapi.web.UserNotFoundException;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DataApiManagerImpl implements DataApiManager {

    private final DataSource sessionSource;
    private final DatabaseLimiter databaseLimiter;
    private final ExternalDatabasesRegistry externalDatabasesRegistry;
    private final CompositeApplicationManager appManager;
    private final AppSettingsRegistry appSettingsRegistry;
    private final MdsSnapshotReferenceManager mdsSnapshotReferenceManager;
    private final ListF<DatabaseChangedEventHandler> snapshotHandlers;
    private final ListF<DatabaseChangedEventAsyncHandler> eventAsyncHandlers;
    private final ExecutorService databaseManagerExecutorService;

    public DataApiManagerImpl(DataSource sessionSource,
            DatabaseLimiter databaseLimiter,
            ExternalDatabasesRegistry externalDatabasesRegistry,
            CompositeApplicationManager appManager,
            AppSettingsRegistry appSettingsRegistry,
            MdsSnapshotReferenceManager mdsSnapshotReferenceManager,
            List<DatabaseChangedEventHandler> snapshotHandlers,
            List<DatabaseChangedEventAsyncHandler> eventAsyncHandlers,
            ExecutorService databaseManagerExecutorService)
    {
        this.sessionSource = sessionSource;
        this.databaseLimiter = databaseLimiter;
        this.externalDatabasesRegistry = externalDatabasesRegistry;
        this.appManager = appManager;
        this.appSettingsRegistry = appSettingsRegistry;
        this.mdsSnapshotReferenceManager = mdsSnapshotReferenceManager;
        this.snapshotHandlers = Cf.x(snapshotHandlers);
        this.eventAsyncHandlers = Cf.x(eventAsyncHandlers);
        this.databaseManagerExecutorService = databaseManagerExecutorService;
    }

    @Override
    public boolean isNoKeepDeltas(UserDatabaseSpec databaseSpec) {
        return sessionSource.type(databaseSpec).hasTrait(DataSourceTrait.NEVER_KEEP_DELTAS)
                || appSettingsRegistry.getDatabaseSettings(databaseSpec.databaseRef()).isNoKeepDeltas();
    }

    private void callDatabaseChangeListenersAsync(DatabaseChange databaseChange) {
        eventAsyncHandlers.stream().map(h ->
                new PriorityRunnable(() -> h.databaseChanged(databaseChange), h.getClass().getSimpleName(), h.getOrder()))
                .forEach(databaseManagerExecutorService::submit);
    }

    private TransactionalManagerSession openSession(UserDatabaseSpec databaseSpec) {
        return openSession(sessionSource, databaseSpec);
    }

    @Override
    public TransactionalManagerSession openSession(DataSource dataSource, UserDatabaseSpec databaseSpec) {
        return new TransactionalManagerSession(dataSource.openSession(databaseSpec));
    }

    public class ManagerSession extends DefaultObject implements DataSourceSessionDecorator {
        final DataSourceSession session;

        ManagerSession(DataSourceSession session) {
            this.session = session;
        }

        @Override
        public DataSourceSession session() {
            return session;
        }

        @Override
        public Database getOrCreateDatabase() {
            validateDatabaseCreation();
            return wrapDatabase(session.getOrCreateDatabase());
        }

        @Override
        public Database createDatabase() {
            validateDatabaseCreation();
            return wrapDatabase(session.createDatabase());
        }

        @Override
        public Option<Database> getDatabaseO() {
            return session.getDatabaseO()
                    .filterMap(this::wrapDatabaseO);
        }

        @Override
        public Option<Snapshot> getSnapshotO(RecordsFilter filter) {
            return session.getSnapshotO(filter)
                    .filter(snapshot -> wrapDatabaseO(snapshot.database).isPresent())
                    .map(s -> s.withDatabaseAlias(databaseAlias()));
        }

        public DeltasAppliedDatabase applyDeltas(Database database, ListF<Delta> deltas, RevisionChecker revChecker) {
            return applyDeltasInner(wrapDatabase(database), deltas, revChecker);
        }

        DeltasAppliedDatabase applyDeltasInner(Database database, ListF<Delta> deltas, RevisionChecker revChecker) {
            revChecker.check(database);
            database.checkCanWrite();

            mdsSnapshotReferenceManager.saveSnapshotToMdsIfNecessary(uid(), database,
                    () -> getSnapshotO(RecordsFilter.DEFAULT)
                            .getOrThrow(databaseRef()::consNotFound)
            );

            SetF<DataRecordId> recordIds = deltas.map(Delta::getChangesRecordIds)
                    .<RecordId>flatten()
                    .map(recordId -> recordId.toDataRecordId(database.dbHandle))
                    .unique();

            Snapshot snapshot = new Snapshot(database, getDataRecordsByIds(recordIds));
            revChecker.check(snapshot, deltas);
            PatchableSnapshot patchableSnapshot = snapshot
                    .toPatchable()
                    .patch(deltas);

            for (DatabaseChangedEventHandler handler : snapshotHandlers) {
                ListF<RecordChange> newChanges = handler.databaseChanged(patchableSnapshot);
                if (newChanges.isEmpty()) {
                    continue;
                }

                patchableSnapshot.patch(new Delta(newChanges).asExtra());
            }

            DatabaseChange change = patchableSnapshot.toDatabaseChange();
            if (isNoKeepDeltas()) {
                change = change.withoutDeltas();
            }

            appManager.validateDatabaseUpdate(change);
            save(change);

            for (Delta delta : change.deltas) {
                DataApiStats.changesInDelta.update(delta.changes.size());
            }

            callDatabaseChangeListenersAsync(change);

            return new DeltasAppliedDatabase(patchableSnapshot.deltas, change);
        }

        boolean isNoKeepDeltas() {
            return DataApiManagerImpl.this.isNoKeepDeltas(databaseSpec());
        }

        Option<SnapshotWithSource> getSnapshotWithRevisionO(Option<Long> rev, RecordsFilter filter) {
            return getSnapshotFromDb(rev, filter).map(SnapshotWithSource::fromDb)
                    .orElse(() -> getDatabaseO().filterMap(
                            db -> getSnapshotFromMdsO(db, rev, filter).map(SnapshotWithSource::fromMds)));
        }

        private Option<Snapshot> getSnapshotFromDb(Option<Long> rev, RecordsFilter filter) {
            return getDatabaseO()
                    .filterMap(db -> rev.exists(r -> r != db.rev)
                            ? Option.empty()
                            : getSnapshotO(filter)
                    );
        }

        private Option<Snapshot> getSnapshotFromMdsO(Database database, Option<Long> revO, RecordsFilter filter) {
            return revO.filterMap(rev -> mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(
                    database.uid, rev, database.dbHandle, filter));
        }

        void deleteDatabaseIfExists(DatabaseDeletionMode deleteMode) {
            if (!getDatabaseO().isPresent()) {
                return;
            }

            deleteDatabase(deleteMode);
        }

        void deleteDatabaseIfExists(DatabaseDeletionMode deleteMode, ChunkRateLimiter rateLimiter) {
            if (!getDatabaseO().isPresent()) {
                return;
            }

            deleteDatabase(deleteMode, rateLimiter);
        }

        /**
         * записываем данные о снапшоте, который нужно сохранить в MDS при любых изменениях данных
         * по mdsKey можно будет потом загрузить знапшот
         * если снапшот уже был создан, то обновляем время последнего запроса
         */
        Database getAndInitCopyOnWrite(Database database) {
            mdsSnapshotReferenceManager.createSnapshotReferenceOrUpdateLastRequestTime(database);
            return database;
        }

        private void validateDatabaseCreation() {
            DatabaseRef databaseRef = databaseRef();
            databaseRef.validate();

            if (databaseAlias().aliasType() != DatabaseAliasType.PUBLIC) {
                appManager.validateDatabaseCreation(uid(), databaseRef);
            }

            switch(databaseAlias().aliasType()) {
                case EXTERNAL:
                    Option<DatabaseAccessType> accessType =
                            externalDatabasesRegistry.getExternalDatabaseAccessType(
                                    (ExternalDatabaseAlias) databaseAlias()
                            );
                    if (!accessType.isSome(DatabaseAccessType.READ_WRITE)) {
                        throw new AccessForbiddenException("Access forbidden for external database " + databaseAlias());
                    }
                    break;

                case PUBLIC:
                    if (!databaseLimiter.isInWhitelist(databaseRef) && !getDatabaseO().isPresent()) {
                        throw new AccessForbiddenException("Access forbidden for public database " + databaseAlias());
                    }
                    break;
            }
        }

        private Database wrapDatabase(Database database) {
            return wrapDatabaseO(database)
                    .getOrThrow(databaseRef()::consNotFound);
        }

        private Option<Database> wrapDatabaseO(Database database) {
            DatabaseRef ref = databaseRef();
            if (ref.hasGlobalContext()) {
                return Option.of(database);
            }

            switch (databaseAlias().aliasType()) {
                case EXTERNAL:
                    return externalDatabasesRegistry.getExternalDatabaseAccessType(
                            (ExternalDatabaseAlias) databaseAlias()
                    ).map(accessType ->
                            database.withAlias(databaseAlias())
                                    .withAccess(accessType)
                    );

                case PUBLIC:
                    return Option.of(
                            databaseSpec().databaseAccess() == DatabaseAccessType.READ_ONLY
                                    ? database.readOnly()
                                    : database
                    );

                default:
                    return Option.of(database);
            }
        }
    }

    public class TransactionalManagerSession extends DefaultObject implements DataSourceSessionDecorator {
        final ManagerSession session;

        TransactionalManagerSession(DataSourceSession session) {
            this.session = new ManagerSession(session);
        }

        @Override
        public DataSourceSession session() {
            return session;
        }

        public DeltasAppliedDatabase applyDeltas(ListF<Delta> deltas, RevisionChecker revisionChecker) {
            return DataApiStats.executeWithProfiling(() ->
                    executeInTxWithLockedDb(db -> session.applyDeltas(db, deltas, revisionChecker))
            );
        }

        @Override
        public Option<Snapshot> getSnapshotO(RecordsFilter filter) {
            return executeInReadOnlyTx(() -> session.getSnapshotO(filter));
        }

        Option<SnapshotWithSource> getSnapshotWithRevisionO(Option<Long> rev, RecordsFilter filter) {
            return executeInReadOnlyTx(() -> session.getSnapshotWithRevisionO(rev, filter));
        }

        void deleteDatabaseIfExists(DatabaseDeletionMode deleteMode) {
            session.deleteDatabaseIfExists(deleteMode);
        }

        void deleteDatabaseIfExists(DatabaseDeletionMode deleteMode, ChunkRateLimiter rateLimiter) {
            session.deleteDatabaseIfExists(deleteMode, rateLimiter);
        }

        Option<Database> getAndInitCopyOnWrite() {
            return tx().executeInTxWithLockedDbIfDbExists(session::getAndInitCopyOnWrite);
        }

        private DeltasAppliedDatabase executeInTxWithLockedDb(Function<Database, DeltasAppliedDatabase> op) {
            return tx().executeInTxWithLockedDb(op);
        }

        private <T> Option<T> executeInReadOnlyTx(Supplier<Option<T>> supplier) {
            try {
                // XXX catch must be somewhere else
                return tx().executeInReadOnlyTx(supplier);
            } catch (UserNotFoundException | DatabaseNotFoundException e) {
                return Option.empty();
            }
        }
    }

    private static final class RecordMatchRevisionChecker extends RevisionChecker {
        private final RecordCondition ifMatch;

        public RecordMatchRevisionChecker(RecordCondition ifMatch) {
            this.ifMatch = ifMatch;
        }

        public void check(Database database) {
            // do nothing
        }

        public void check(Snapshot snapshot, ListF<Delta> deltas) {
            Option<DataRecord> mismatching = snapshot.records.records().find(ifMatch.not()::matches);

            if (mismatching.isPresent()) {
                throw new OutdatedChangeException(
                        "Record " + mismatching.get().id().idStr() + " does not match specified condition");
            }
        }
    }

    public static abstract class RevisionChecker extends DefaultObject {

        public static RevisionChecker cons(RevisionCheckMode mode, long rev) {
            return new RevisionCheckerImpl(mode, rev);
        }

        public static RevisionChecker ifMatch(RecordCondition condition) {
            return new RecordMatchRevisionChecker(condition);
        }

        public abstract void check(Database database);

        public abstract void check(Snapshot snapshot, ListF<Delta> deltas);

        public boolean isRecordMatcher() {
            return this instanceof RecordMatchRevisionChecker;
        }

        private static class RevisionCheckerImpl extends RevisionChecker {
            final RevisionCheckMode mode;
            final long rev;

            RevisionCheckerImpl(RevisionCheckMode mode, long rev) {
                this.mode = mode;
                this.rev = rev;
            }

            @Override
            public void check(Database database) {
                if (database.rev > rev && RevisionCheckMode.WHOLE_DATABASE == mode) {
                    throw new OutdatedChangeException(database.rev, rev);
                }

                if (database.rev < rev) {
                    throw new TooNewChangeException(database.rev, rev);
                }
            }

            @Override
            public void check(Snapshot snapshot, ListF<Delta> deltas) {
                Database database = snapshot.database;
                CollectionF<DataRecord> records = snapshot.records();

                if (mode != RevisionCheckMode.PER_RECORD) {
                    return;
                }

                if (rev == database.rev) {
                    return;
                }

                checkRecordRevisions(deltas, records);
                ensureAllRecordsExist(deltas, database, records);
            }

            private void checkRecordRevisions(ListF<Delta> deltas, CollectionF<DataRecord> records) {
                SetF<SimpleRecordId> forced = Cf.hashSet();

                deltas.reverseIterator().flatMap(d -> d.changes.reverseIterator()).forEachRemaining(change -> {
                    if (change.forced && change.type == RecordChangeType.SET) {
                        forced.add(new SimpleRecordId(change.collectionId, change.recordId));
                    } else {
                        forced.removeTs(new SimpleRecordId(change.collectionId, change.recordId));
                    }
                });

                Option<DataRecord> conflictO = records.find(r -> r.rev > rev && !forced.containsTs(r.simpleId()));
                if (conflictO.isPresent()) {
                    DataRecord conflict = conflictO.get();
                    String id = String.format("'%s'", conflict.id().idStr());
                    throw new OutdatedChangeException(id, conflict.rev, rev);
                }
            }

            private void ensureAllRecordsExist(ListF<Delta> deltas, Database database,
                    CollectionF<DataRecord> records)
            {
                Option<RecordChange> changeO = DeltaUtils.findNonExistingRecordDeletionOrUpdate(database.dbHandle,
                        records, deltas);
                if (changeO.isPresent()) {
                    RecordChange change = changeO.get();
                    // prevent DeltaValidationException with 400 status code
                    throw new OutdatedChangeException("Can't "
                            + (change.type == RecordChangeType.DELETE ? "delete" : "update")
                            + " non-existing record '" + change.getRecordId() + "'");
                }
            }
        }
    }

    @Override
    public Option<Database> getAndInitCopyOnWrite(UserDatabaseSpec databaseSpec) {
        return openSession(databaseSpec)
                .getAndInitCopyOnWrite();
    }


    // databases

    @Override
    public Database getDatabase(UserDatabaseSpec databaseSpec) {
        return openSession(databaseSpec)
                .getDatabase();
    }

    @Override
    public Database getOrCreateDatabase(UserDatabaseSpec databaseSpec) {
        return openSession(databaseSpec)
                .getOrCreateDatabase();
    }

    @Override
    public void runWithLockedDatabase(UserDatabaseSpec databaseSpec, Function1V<LockedDatabaseSession> function) {
        executeWithLockedDatabase(databaseSpec, function.asFunctionReturnValue(true));
    }

    @Override
    public <R> R executeWithLockedDatabase(UserDatabaseSpec databaseSpec, Function<LockedDatabaseSession, R> function) {
        DsSessionTxManager session = openSession(databaseSpec).tx();

        return session.executeInTxWithLockedDb(
                db -> function.apply(new LockedDatabaseSession(db, new ManagerSession(session.session()))));
    }

    @Override
    public Database createDatabase(UserDatabaseSpec databaseSpec) {
        return openSession(databaseSpec)
                .createDatabase();
    }

    @Override
    public Option<Database> getDatabaseO(UserDatabaseSpec databaseSpec) {
        return openSession(databaseSpec)
                .getDatabaseO();
    }

    @Override
    public void deleteDatabaseIfExists(UserDatabaseSpec databaseSpec, DatabaseDeletionMode deleteMode) {
        openSession(databaseSpec)
                .deleteDatabaseIfExists(deleteMode);
    }

    @Override
    public void deleteDatabaseIfExists(UserDatabaseSpec databaseSpec, DatabaseDeletionMode deleteMode,
                                       ChunkRateLimiter rateLimiter) {
        openSession(databaseSpec)
                .deleteDatabaseIfExists(deleteMode, rateLimiter);
    }

    @Override
    public Database setDatabaseDescription(UserDatabaseSpec databaseSpec, Option<String> newTitle) {
        return openSession(databaseSpec)
                .setDatabaseDescription(newTitle);
    }

    @Override
    public Database createDatabaseWithDescription(UserDatabaseSpec databaseSpec, String title) {
        return openSession(databaseSpec)
                .createDatabaseWithDescription(title);
    }

    @Override
    public void onDatabaseUpdate(UserDatabaseSpec databaseSpec, long rev) {
        openSession(databaseSpec)
                .onDatabaseUpdate(rev);
    }


    // deltas

    @Override
    public Delta getDelta(UserDatabaseSpec databaseSpec, long rev) {
        return openSession(databaseSpec)
                .getDelta(rev);
    }

    @Override
    public ListF<Delta> listDeltas(UserDatabaseSpec databaseSpec, long fromRev, int limit) {
        return openSession(databaseSpec)
                .listDeltas(fromRev, limit);
    }

    @Override
    public Database applyDelta(Database database, RecordChange change, RecordCondition ifMatch) {
        return applyDeltas(UserDatabaseSpec.fromDatabase(database),
                Cf.list(new Delta(change)), new RecordMatchRevisionChecker(ifMatch)).database;
    }

    @Override
    public Database applyDelta(Database database, RevisionCheckMode revCheckMode, Delta delta) {
        return applyDelta(UserDatabaseSpec.fromDatabase(database), database.rev, revCheckMode, delta);
    }

    @Override
    public Database applyDelta(UserDatabaseSpec databaseSpec, long rev, RevisionCheckMode revCheckMode,
                               Delta delta)
    {
        return applyDeltas(databaseSpec, rev, revCheckMode, Cf.list(delta)).database;
    }

    @Override
    public DeltasAppliedDatabase applyDeltas(UserDatabaseSpec databaseSpec, long rev,
                                             RevisionCheckMode revCheckMode, ListF<Delta> deltas) {
        return applyDeltas(databaseSpec, deltas, RevisionChecker.cons(revCheckMode, rev));
    }

    private DeltasAppliedDatabase applyDeltas(
            UserDatabaseSpec databaseSpec, ListF<Delta> deltas, RevisionChecker revChecker)
    {
        Function0<DeltasAppliedDatabase> deltasApplier =
                () -> openSession(databaseSpec).applyDeltas(deltas, revChecker);
        try {
            return deltasApplier.apply();

        } catch (StuckBehindDatabaseException e) {
            try {
                openSession(databaseSpec).fixDatabaseRevision(e.getDatabase().rev + 1, e.getDatabase().rev);

            } catch (DatabaseRevisionMismatchException ignored) {}

            if (revChecker.isRecordMatcher()) {
                return deltasApplier.apply();
            }
            throw new OutdatedChangeException(StringUtils.format(
                    "Requested update with rev={} for stuck behind database={}",
                    e.getDatabase().rev, e.getDatabase().handleValue()));
        }
    }


    // snapshots

    @Override
    public Snapshot getSnapshot(UserDatabaseSpec databaseSpec, RecordsFilter filter, SnapshotSource source) {
        return getSnapshotO(databaseSpec, filter, source)
                    .getOrThrow(databaseSpec::consNotFound);
    }

    @Override
    public Option<Snapshot> getSnapshotO(UserDatabaseSpec databaseSpec, RecordsFilter filter, SnapshotSource source) {
        Function0<Option<Snapshot>> loadSnapshot = () -> openSession(databaseSpec).getSnapshotO(filter);

        if (source == SnapshotSource.BY_CURRENT_POLICY || sessionSource.type(databaseSpec) != DataSourceType.DISK) {
            return loadSnapshot.apply();
        }

        Option<Snapshot> slaveSnapshot = MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_SM, loadSnapshot);
        Option<Database> slaveDb = slaveSnapshot.map(s -> s.database);

        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_M, () -> {
            Option<Database> masterDb = openSession(databaseSpec.withoutDatabase()).getDatabaseO();

            if (!masterDb.isPresent()) {
                return Option.empty();
            }
            if (!slaveDb.exists(db -> masterDb.get().handleValue().equals(db.handleValue()))) {
                return loadSnapshot.apply();
            }
            if (slaveDb.get().rev == masterDb.get().rev) {
                return slaveSnapshot;
            }
            if (source == SnapshotSource.SLAVE_THEN_MASTER || !filter.limits().isAll()) {
                return loadSnapshot.apply();
            }
            ListF<Delta> deltas = openSession(masterDb.get().spec()).listDeltas(
                    slaveDb.get().rev, (int) (masterDb.get().rev - slaveDb.get().rev));

            if (deltas.isEmpty() || !deltas.forAll(new DeltaUtils.RevSequenceChecker(slaveDb.get().rev)::matches)) {
                return loadSnapshot.apply();
            }
            SetF<String> fields = filter.getRecordCond().getDataFields();

            if (fields.isNotEmpty() && deltas.exists(d -> d.updatesAnyField(fields))) {
                return loadSnapshot.apply();
            }

            ListF<DataRecord> records = slaveSnapshot.get()
                    .toPatchable(Option.of(filter.getRecordOrder()).filterByType(ByIdRecordOrder.class))
                    .withValidationMode(DeltaValidationMode.IGNORE_NON_EXISTENT)
                    .patch(deltas).patchedRecords.records().toList();

            if (filter.hasAnyCondition()) {
                records = records.filter(filter::matches);
            }
            if (!(filter.getRecordOrder() instanceof ByIdRecordOrder)) {
                records = records.sorted(filter.getRecordOrder().comparator());
            }
            return Option.of(new Snapshot(masterDb.get(), records));
        });
    }

    @Override
    public Option<SnapshotWithSource> getSnapshotWithRevisionO(
            UserDatabaseSpec dbSpec, Option<Long> rev, RecordsFilter filter)
    {
        return openSession(dbSpec)
                    .getSnapshotWithRevisionO(rev, filter);
    }

    // admin-only
    @Override
    public Snapshot getSnapshot(UserDatabaseSpec databaseSpec) {
        return getSnapshot(databaseSpec, RecordsFilter.DEFAULT);
    }

    // internal API
    @Override
    public SnapshotPojoRow getSnapshotRow(DataApiUserId uid, RecordRef recordId) {
        return getRecord(uid, recordId)
                .map(DataRecord::toSnapshotPojoRow)
                .getOrThrow(NotFoundException.consF("Record not found"));
    }


    // records

    @Override
    public Option<DataRecord> getRecord(DataApiUserId uid, RecordRef recordRef) {
        return getRecord(new UserDatabaseSpec(uid, recordRef.dbRef()), recordRef);
    }

    @Override
    public Option<DataRecord> getRecord(UserDatabaseSpec databaseSpec, RecordId recordId) {
        return getRecords(databaseSpec, RecordsFilter.DEFAULT.withRecordRef(recordId)).singleO();
    }

    @Override
    public ListF<DataRecord> getRecords(UserDatabaseSpec databaseSpec) {
        return getRecords(databaseSpec, RecordsFilter.DEFAULT);
    }

    @Override
    public ListF<DataRecord> getRecords(UserDatabaseSpec databaseSpec, RecordsFilter filter) {
        return openSession(databaseSpec)
                .getDataRecords(filter);
    }

    @Override
    public int getRecordsCount(UserDatabaseSpec databaseSpec) {
        return getRecordsCount(databaseSpec, RecordsFilter.DEFAULT);
    }

    @Override
    public int getRecordsCount(UserDatabaseSpec databaseSpec, RecordsFilter filter) {
        return openSession(databaseSpec)
                .getDataRecordsCount(filter);
    }

    @Override
    public ListF<Snapshot> takeout(DataApiUserId userId, ListF<Database> databases) {
        return databases.map(db -> getSnapshot(UserDatabaseSpec.fromDatabase(db)));
    }
}
