package ru.yandex.chemodan.app.dataapi.core.manager;

import java.util.concurrent.ExecutorService;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataColumn;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSource;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSession;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.datasource.NopSessionTxManager;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.CompositeApplicationManager;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.limiter.DatabaseLimiter;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceManager;
import ru.yandex.chemodan.app.dataapi.support.RecordField;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
@RunWith(MockitoJUnitRunner.class)
public class DataApiManagerSnapshotTest {
    @Mock
    private DataSourceSession masterSession;
    @Mock
    private DataSourceSession slaveSession;

    private DataApiManagerImpl manager;

    private final DataApiUserId uid = new DataApiPassportUserId(1L);
    private final DatabaseHandle handle = DatabaseHandle.consGlobal("dbId", "handle");
    private final UserDatabaseSpec spec = UserDatabaseSpec.fromUserAndHandle(uid, handle);

    @Before
    public void setup() {
        DataSource dataSource = Mockito.mock(DataSource.class);

        Mockito.when(dataSource.type(Mockito.any())).thenReturn(DataSourceType.DISK);
        Mockito.when(dataSource.openSession(Mockito.any())).then(i ->
                MasterSlaveContextHolder.policy() == MasterSlavePolicy.R_M ? masterSession : slaveSession);

        Cf.list(masterSession, slaveSession).forEach(session -> {
            Mockito.when(session.tx()).thenReturn(new NopSessionTxManager(session));
            Mockito.when(session.databaseSpec()).thenReturn(spec);
        });

        manager = new DataApiManagerImpl(dataSource,
                Mockito.mock(DatabaseLimiter.class), Mockito.mock(ExternalDatabasesRegistry.class),
                Mockito.mock(CompositeApplicationManager.class), Mockito.mock(AppSettingsRegistry.class),
                Mockito.mock(MdsSnapshotReferenceManager.class),
                Cf.list(), Cf.list(), Mockito.mock(ExecutorService.class));
    }

    @Test
    public void gotFromSlave() {
        Snapshot snapshot = new Snapshot(Database.consNew(uid, handle), Cf.list());

        Mockito.when(slaveSession.getSnapshotO(Mockito.any())).thenReturn(Option.of(snapshot));
        Mockito.when(masterSession.getDatabaseO()).thenReturn(Option.of(snapshot.database));

        Assert.some(snapshot, manager.getSnapshotO(spec, RecordsFilter.DEFAULT, SnapshotSource.SLAVE_THEN_DELTAS));
    }

    @Test
    public void gotFromMaster() {
        Snapshot slaveSnapshot = new Snapshot(Database.consNew(uid, handle), Cf.list());
        Snapshot masterSnapshot = new Snapshot(slaveSnapshot.database.withRev(1), Cf.list());

        Mockito.when(slaveSession.getSnapshotO(Mockito.any())).thenReturn(Option.of(slaveSnapshot));
        Mockito.when(masterSession.getDatabaseO()).thenReturn(Option.of(masterSnapshot.database));
        Mockito.when(masterSession.getSnapshotO(Mockito.any())).thenReturn(Option.of(masterSnapshot));

        Assert.some(masterSnapshot, manager.getSnapshotO(spec, RecordsFilter.DEFAULT, SnapshotSource.SLAVE_THEN_MASTER));
    }

    @Test
    public void patchedFilteredAndSorted() {
        Delta delta1 = new Delta(Cf.list(
                RecordChange.insert("col", "x", Cf.map("ord", DataField.integer(2))),
                RecordChange.insert("col", "y", Cf.map("ord", DataField.integer(1))),
                RecordChange.delete("missing", "x"))).withRev(0);

        Delta delta2 = new Delta(Cf.list(
                RecordChange.update("col", "y", Cf.list(FieldChange.put("ord", DataField.integer(3)))),
                RecordChange.insert("col", "z", Cf.map("ord", DataField.integer(1))),
                RecordChange.update("missing", "y", Cf.list()))).withRev(1);

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionId("col")
                .withRecordOrder(RecordField.intNumber("ord").column().orderBy());

        Snapshot slaveSnapshot = new Snapshot(Database.consNew(uid, handle), Cf.list());
        Snapshot masterSnapshot = new Snapshot(slaveSnapshot.database.withRev(2), Cf.list(
                new DataRecord(uid, new DataRecordId(handle, "col", "z"), 2, Cf.map("ord", DataField.integer(1))),
                new DataRecord(uid, new DataRecordId(handle, "col", "x"), 1, Cf.map("ord", DataField.integer(2))),
                new DataRecord(uid, new DataRecordId(handle, "col", "y"), 2, Cf.map("ord", DataField.integer(3)))));

        Mockito.when(slaveSession.getSnapshotO(Mockito.any())).thenReturn(Option.of(slaveSnapshot));
        Mockito.when(masterSession.getDatabaseO()).thenReturn(Option.of(masterSnapshot.database));
        Mockito.when(masterSession.listDeltas(0, 2)).thenReturn(Cf.list(delta1, delta2));

        Assert.some(masterSnapshot, manager.getSnapshotO(spec, filter, SnapshotSource.SLAVE_THEN_DELTAS));
    }

    @Test
    public void deltaGone() {
        Snapshot slaveSnapshot = new Snapshot(Database.consNew(uid, handle), Cf.list());
        Snapshot masterSnapshot = new Snapshot(slaveSnapshot.database.withRev(1), Cf.list());

        Delta delta = new Delta(RecordChange.insert(new SimpleRecordId("x", "x"), Cf.map())).withRev(1);

        Mockito.when(slaveSession.getSnapshotO(Mockito.any())).thenReturn(Option.of(slaveSnapshot));
        Mockito.when(masterSession.getDatabaseO()).thenReturn(Option.of(masterSnapshot.database));
        Mockito.when(masterSession.getSnapshotO(Mockito.any())).thenReturn(Option.of(masterSnapshot));

        Mockito.when(masterSession.listDeltas(Mockito.anyLong(), Mockito.anyInt())).thenReturn(Cf.list());
        Assert.some(masterSnapshot, manager.getSnapshotO(spec, RecordsFilter.DEFAULT, SnapshotSource.SLAVE_THEN_DELTAS));

        Mockito.when(masterSession.listDeltas(Mockito.anyLong(), Mockito.anyInt())).thenReturn(Cf.list(delta));
        Assert.some(masterSnapshot, manager.getSnapshotO(spec, RecordsFilter.DEFAULT, SnapshotSource.SLAVE_THEN_DELTAS));
    }

    @Test
    public void deltaUpdatesConditionField() {
        Snapshot slaveSnapshot = new Snapshot(Database.consNew(uid, handle), Cf.list());
        Snapshot masterSnapshot = new Snapshot(slaveSnapshot.database.withRev(1), Cf.list());

        Delta delta = new Delta(Cf.list(
                RecordChange.insert("x", "x", Cf.map()),
                RecordChange.update("y", "y", Cf.list(FieldChange.delete("cond"))))).withRev(0);

        RecordsFilter filter = RecordsFilter.DEFAULT.withDataCond(DataColumn.string("cond").eq("value"));

        Mockito.when(slaveSession.getSnapshotO(Mockito.any())).thenReturn(Option.of(slaveSnapshot));
        Mockito.when(masterSession.getDatabaseO()).thenReturn(Option.of(masterSnapshot.database));

        Mockito.when(masterSession.listDeltas(Mockito.anyLong(), Mockito.anyInt())).thenReturn(Cf.list(delta));
        Mockito.when(masterSession.getSnapshotO(Mockito.any())).thenReturn(Option.of(masterSnapshot));

        Assert.some(masterSnapshot, manager.getSnapshotO(spec, filter, SnapshotSource.SLAVE_THEN_DELTAS));
    }
}
