package ru.yandex.chemodan.app.dataapi.core.manager;

import java.util.concurrent.atomic.AtomicReference;

import org.joda.time.Duration;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.dataapi.api.DatabaseChangedEventAsyncHandler;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.dataapi.test.ManualResetEvent;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@ActivateDataApiEmbeddedPg
public class DataApiManagerTest extends DataApiTestSupport {
    private DataApiUserId uid;

    @Before
    public void Before() {
        timeStop();
        uid = createRandomCleanUserInDefaultShard();
    }

    @Test
    public void callListenerTest() throws Exception {
        ManualResetEvent mre = new ManualResetEvent(false);
        final AtomicReference<DatabaseChange> actualEvent = new AtomicReference<>();

        registerDependency((DatabaseChangedEventAsyncHandler) event -> {
            actualEvent.set(event);
            mre.set();
        });

        //snapshotReferenceDeletionInterval for DatabaseManager
        registerDependency(Duration.standardDays(1000));

        MapF<String, DataField> startData = Cf.map("field1", DataField.decimal(10));
        MapF<String, DataField> endData = Cf.map("field2", DataField.string("1212"));

        AppDatabaseRef databaseRef = new AppDatabaseRef("app", "database1");
        UserDatabaseSpec databaseSpec = new UserDatabaseSpec(uid, databaseRef);

        DataApiManagerImpl dataApiManager = createBean(DataApiManagerImpl.class);
        dataApiManager.createDatabase(databaseSpec);
        dataApiManager.applyDeltas(databaseSpec, 0, RevisionCheckMode.PER_RECORD,
                Cf.list(new Delta(RecordChange.insert("collection-1", "record-1", startData)))
        );

        Assert.isTrue(mre.waitOne(30 * 1000));
        mre.reset();

        Assert.sizeIs(1, actualEvent.get().deltas);
        Assert.equals(1L, actualEvent.get().patchedDatabase().rev);
        Assert.equals(databaseRef.appNameO(), actualEvent.get().patchedDatabase().appNameO());
        Assert.equals(uid, actualEvent.get().patchedDatabase().uid);
        Assert.equals(databaseRef.databaseId(), actualEvent.get().patchedDatabase().databaseId());
        Assert.sizeIs(0, actualEvent.get().getSourceRecords());
        Assert.sizeIs(1, actualEvent.get().getPatchedRecords());
        Assert.equals(startData, actualEvent.get().getPatchedRecords().iterator().next().getData());

        dataApiManager.applyDeltas(databaseSpec, 1, RevisionCheckMode.PER_RECORD,
                Cf.list(new Delta(RecordChange.set("collection-1", "record-1", endData)))
        );

        Assert.isTrue(mre.waitOne(30 * 1000));

        Assert.sizeIs(1, actualEvent.get().deltas);
        Assert.equals(2L, actualEvent.get().patchedDatabase().rev);
        Assert.equals(databaseRef.appNameO(), actualEvent.get().patchedDatabase().appNameO());
        Assert.equals(uid, actualEvent.get().patchedDatabase().uid);
        Assert.equals(databaseRef.databaseId(), actualEvent.get().patchedDatabase().databaseId());

        Assert.sizeIs(1, actualEvent.get().getSourceRecords());
        Assert.sizeIs(1, actualEvent.get().getPatchedRecords());

        Assert.equals(startData, actualEvent.get().getSourceRecords().iterator().next().getData());
        Assert.equals(endData, actualEvent.get().getPatchedRecords().iterator().next().getData());
    }
}
