package ru.yandex.chemodan.app.dataapi.core.manager;

import java.util.concurrent.Executors;

import org.junit.Test;
import org.mockito.stubbing.Answer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecords;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSource;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSession;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseAccessType;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.db.ref.internalpublic.PublicDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPublicUserId;
import ru.yandex.chemodan.app.dataapi.apps.CompositeApplicationManager;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppDatabaseSettings;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.limiter.DatabaseLimiter;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceManager;
import ru.yandex.chemodan.app.dataapi.test.TestConstants;
import ru.yandex.chemodan.app.dataapi.web.AccessForbiddenException;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DataApiManagerUnitTest {
    private static final DataApiPassportUserId TEST_UID = new DataApiPassportUserId(1L);

    private static final ExternalDatabaseAlias EXT_DB_ALIAS_RW = TestConstants.EXT_DB_ALIAS_RW;

    private static final PublicDatabaseAlias PUBLIC_ALIAS = new PublicDatabaseAlias("app", "dbId");

    @Test
    public void externalDbIsWrappedOnCreate() {
        Database database = consDataApiManager()
                .createDatabase(UserDatabaseSpec.fromUserAndAlias(TEST_UID, EXT_DB_ALIAS_RW));
        Assert.equals(EXT_DB_ALIAS_RW, database.alias);
    }

    @Test
    public void externalDbIsWrappedOnGetSnapshot() {
        Database database = consDataApiManager()
                .getSnapshot(UserDatabaseSpec.fromUserAndAlias(TEST_UID, EXT_DB_ALIAS_RW))
                .database;
        Assert.equals(EXT_DB_ALIAS_RW, database.alias);
    }

    @Test
    public void publicDbIsNotReadOnlyForPublicUser() {
        // AccessForbiddenException MUST not be thrown
        applySomeDelta(UserDatabaseSpec.fromUserAndAlias(new DataApiPublicUserId(), PUBLIC_ALIAS));
    }

    @Test(expected = AccessForbiddenException.class)
    public void publicDbIsReadOnlyForNonPublicUser() {
        applySomeDelta(UserDatabaseSpec.fromUserAndAlias(new DataApiPassportUserId(123), PUBLIC_ALIAS));
    }

    @SuppressWarnings("UnusedReturnValue")
    private static Database applySomeDelta(UserDatabaseSpec databaseSpec) {
        DataApiManager dataApiManager = consDataApiManager();
        Database database = dataApiManager.getOrCreateDatabase(databaseSpec);
        return dataApiManager.applyDelta(
                database, RevisionCheckMode.WHOLE_DATABASE,
                new SimpleDataRecord("colId", "recId", DataFields.EMPTY)
                        .toInsertChange()
                        .toDelta()
        );
    }

    private static DataApiManager consDataApiManager() {
        ExternalDatabasesRegistry extDbRegistry = mock(ExternalDatabasesRegistry.class);
        when(extDbRegistry.getExternalDatabaseAccessType(EXT_DB_ALIAS_RW))
                .thenReturn(Option.of(DatabaseAccessType.READ_WRITE));

        DatabaseLimiter databaseLimiter = mock(DatabaseLimiter.class);
        when(databaseLimiter.isInWhitelist(PUBLIC_ALIAS.dbRef()))
                .thenReturn(true);

        AppSettingsRegistry mock = mock(AppSettingsRegistry.class);
        when(mock.getDatabaseSettings(any()))
                .thenAnswer(
                        (Answer<AppDatabaseSettings>) inv ->
                                new AppDatabaseSettings((DatabaseRef) inv.getArguments()[0])
                );

        return new DataApiManagerImpl(
                consTestDataSource(),
                databaseLimiter,
                extDbRegistry,
                mock(CompositeApplicationManager.class),
                mock,
                mock(MdsSnapshotReferenceManager.class),
                Cf.list(),
                Cf.list(),
                Executors.newSingleThreadExecutor()
        );
    }

    private static DataSource consTestDataSource() {
        return new DataSource() {
            @Override
            public DataSourceType type(UserDatabaseSpec spec) {
                return DataSourceType.DISK;
            }

            @Override
            public DataSourceSession openSession(UserDatabaseSpec ds) {
                return new DataSourceSession() {
                    @Override
                    public UserDatabaseSpec databaseSpec() {
                        return ds;
                    }

                    @Override
                    public Database getOrCreateDatabase() {
                        return createDatabase();
                    }

                    @Override
                    public Database createDatabase() {
                        return Database.consNew(ds.uid(), ds.databaseRef().consHandle("123"));
                    }

                    @Override
                    public Option<Database> getDatabaseO() {
                        return Option.of(createDatabase());
                    }

                    @Override
                    public void deleteDatabase(DatabaseDeletionMode deletionMode) {

                    }

                    @Override
                    public void deleteDatabase(DatabaseDeletionMode deletionMode, ChunkRateLimiter rateLimiter) {

                    }

                    @Override
                    public Database setDatabaseDescription(Option<String> newTitle) {
                        return createDatabase()
                                .withDescription(newTitle);
                    }

                    @Override
                    public Database createDatabaseWithDescription(String title) {
                        return setDatabaseDescription(Option.of(title));
                    }

                    @Override
                    public Database fixDatabaseRevision(long rev, long currentRev) {
                        return createDatabase();
                    }

                    @Override
                    public void onDatabaseUpdate(long rev) {

                    }

                    @Override
                    public Delta getDelta(long rev) {
                        return null;
                    }

                    @Override
                    public ListF<Delta> listDeltas(long fromRev, int limit) {
                        return Cf.list();
                    }

                    @Override
                    public Option<Snapshot> getSnapshotO(RecordsFilter filter) {
                        return Option.of(new Snapshot(createDatabase(), DataRecords.EMPTY));
                    }

                    @Override
                    public Option<ListF<DataRecord>> getDataRecordsO(RecordsFilter filter) {
                        return Option.empty();
                    }

                    @Override
                    public ListF<DataRecord> getDataRecordsByIds(SetF<DataRecordId> recordIds) {
                        return Cf.list();
                    }

                    @Override
                    public int getDataRecordsCount(RecordsFilter filter) {
                        return 0;
                    }

                    @Override
                    public void save(DatabaseChange change) {

                    }
                };
            }
        };
    }
}
