package ru.yandex.chemodan.app.dataapi.core.manager;

import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.internalpublic.PublicDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPublicUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.limiter.DatabaseLimiter;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.dataapi.web.AccessForbiddenException;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@ActivateDataApiEmbeddedPg
public class PublicDatabaseDataApiManagerTest extends DataApiTestSupport {
    private static final PublicDatabaseAlias PUBLIC_ALIAS = new PublicDatabaseAlias("app", "dbId");

    @Autowired
    private DataApiManager dataApiManager;

    @Autowired
    private DatabaseLimiter databaseLimiter;

    @Before
    public void before() {
        userInitializer.initUserForTests(new DataApiPublicUserId(), false);
    }

    @Test
    public void publicDbIsNotReadOnlyForPublicUser() {
        // AccessForbiddenException MUST not be thrown
        applyDelta(UserDatabaseSpec.fromUserAndAlias(new DataApiPublicUserId(), PUBLIC_ALIAS));
    }

    @Test(expected = AccessForbiddenException.class)
    public void publicDbIsReadOnlyForNonPublicUser() {
        applyDelta(UserDatabaseSpec.fromUserAndAlias(new DataApiPassportUserId(123), PUBLIC_ALIAS));
    }

    private void applyDelta(UserDatabaseSpec databaseSpec) {
        databaseLimiter.applicationWhitelist.get().add("app");
        Database database = dataApiManager.getOrCreateDatabase(databaseSpec);

        dataApiManager.applyDelta(
                database, RevisionCheckMode.WHOLE_DATABASE,
                new SimpleDataRecord("colId", "recId", DataFields.EMPTY)
                        .toInsertChange()
                        .toDelta()
        );
    }

    @Test
    public void getOrCreateIsWithoutCreateRightsOkWhenDatabaseExists() {
        // create database
        databaseLimiter.applicationWhitelist.get().add("app");
        UserDatabaseSpec databaseSpec = UserDatabaseSpec.fromUserAndAlias(new DataApiPublicUserId(), PUBLIC_ALIAS);
        dataApiManager.createDatabase(databaseSpec);

        databaseLimiter.applicationWhitelist.get().removeTs("app");
        // must not throw if database exists
        dataApiManager.getOrCreateDatabase(databaseSpec);
    }
}
