package ru.yandex.chemodan.app.dataapi.core.mdssnapshot;

import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseMeta;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.test.Assert;

/**
 * @author osidorkin
 */
public class MdsSnapshotProcessorTest {
    public static final DataApiPassportUserId TEST_UID = new DataApiPassportUserId(1L);
    public static final DatabaseHandle DATABASE_HANDLE = new DatabaseHandle("test_app", "test_db", "test_handle");

    @Test
    public void applyFilteringAndLimits() {
        Instant instant = Instant.now();
        Snapshot snapshot = generateSnapshot(instant);

        Snapshot newSnapshot = applyFilteringAndLimits(snapshot, Option.empty(),
                ByIdRecordOrder.COLLECTION_ID_ASC_RECORD_ID_ASC, SqlLimits.all());
        Assert.equals(snapshot, newSnapshot);

        newSnapshot = applyFilteringAndLimits(snapshot, Option.of(Cf.set("1", "2", "4")),
                ByIdRecordOrder.COLLECTION_ID_ASC_RECORD_ID_ASC, SqlLimits.all());
        Assert.equals(fromRowsSnapshot(
                        generateRows("1", 10).plus(generateRows("2", 2)).plus(generateRows("4", 2)),
                        instant),
                        newSnapshot);

        newSnapshot = applyFilteringAndLimits(snapshot, Option.of(Cf.set("1", "2", "4")),
                ByIdRecordOrder.COLLECTION_ID_ASC_RECORD_ID_ASC, SqlLimits.range(3, 5));
        Assert.sizeIs(5, newSnapshot.records());
        Assert.assertListsEqual(snapshot.records().toList().subList(3, 8), newSnapshot.records().toList());


        ListF<DataRecord> rows = applyFilteringAndLimits(snapshot, Option.empty(),
                ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC, SqlLimits.all())
                .records()
                .toList();
        Assert.assertListsEqual(snapshot.records().toList().reverse(), rows);

        rows = applyFilteringAndLimits(snapshot, Option.of(Cf.set("1", "2", "4")),
                ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC, SqlLimits.all())
                .records()
                .toList();
        Assert.equals(generateRows("1", 10).plus(generateRows("2", 2)).plus(generateRows("4", 2)).reverse(), rows);

        rows = applyFilteringAndLimits(snapshot, Option.of(Cf.set("1", "2", "4")),
                ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC, SqlLimits.range(0, 4))
                .records()
                .toList();
        Assert.equals(generateRows("2", 2).plus(generateRows("4", 2)).reverse(), rows);

        rows = applyFilteringAndLimits(snapshot, Option.of(Cf.set("1", "2", "4")),
                ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC, SqlLimits.range(snapshot.recordCount() + 1, 4))
                .records()
                .toList();
        Assert.equals(Cf.list(), rows);
    }

    private static Snapshot applyFilteringAndLimits(Snapshot snapshot, Option<SetF<String>> collectionIdsO,
            ByIdRecordOrder recordOrder, SqlLimits limits)
    {
        return MdsSnapshotProcessor.applyFilteringAndLimits(snapshot,
                RecordsFilter.DEFAULT
                        .withCollectionIdsO(collectionIdsO)
                        .withRecordOrder(recordOrder)
                        .withLimits(limits)
        );
    }

    public static Snapshot generateSnapshot(Instant instant) {
        ListF<DataRecord> rows = Cf.<DataRecord>list()
            .plus(generateRows("1", 10))
            .plus(generateRows("2", 2))
            .plus(generateRows("3", 2))
            .plus(generateRows("4", 2))
            .plus(generateRows("5", 2));
        return fromRowsSnapshot(rows, instant);
    }

    public static Snapshot fromRowsSnapshot(ListF<DataRecord> rows, Instant instant) {
        DatabaseMeta meta = new DatabaseMeta(
                instant.minus(Duration.standardDays(3)),
                instant.minus(Duration.standardDays(1)),
                DataSize.fromBytes(100), 18);
        Database database = new Database(TEST_UID, DATABASE_HANDLE, 1L, meta);
        return new Snapshot(database, rows);

    }
    public static ListF<DataRecord> generateRows(String collectionId, int recordCount) {
        return IntStream.range(0, recordCount)
                .mapToObj(i -> generateRecord(collectionId, i))
                .collect(Collectors.toCollection(Cf::arrayList));
    }

    private static DataRecord generateRecord(String collectionId, int i) {
        DataRecordId recordId = new DataRecordId(DATABASE_HANDLE, collectionId, String.valueOf(i));
        return new DataRecord(TEST_UID, recordId, 1L, DataFields.EMPTY);
    }
}
