package ru.yandex.chemodan.app.dataapi.core.mdssnapshot;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.dataapi.test.DataCleaner;
import ru.yandex.chemodan.app.dataapi.utils.elliptics.EllipticsHelper;
import ru.yandex.inside.elliptics.EllipticsFileNotFoundException;
import ru.yandex.inside.elliptics.EllipticsUploadState;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class MdsSnapshotReferenceManagerTest extends DataApiTestSupport {

    @Autowired
    private MdsSnapshotReferenceJdbcDao mdsSnapshotReferenceJdbcDao;

    @Autowired
    private MdsSnapshotReferenceManager mdsSnapshotReferenceManager;

    @Autowired
    private DataCleaner dataCleaner;

    @Autowired
    private EllipticsHelper ellipticsHelper;

    @Autowired
    private DataApiManager dataApiManager;

    private DataApiUserId uid;

    @Before
    public void Before() {
        uid = createRandomCleanUserInDefaultShard();
        dataCleaner.deleteAllSnapshotReferences(uid);
    }

    @Test
    public void deleteOldSnapshotReferences_HaveYoungAndOldSnapshotReferences_DeletesOnlyOld() {
        String handle = Random2.R.nextAlnum(30);

        MdsSnapshotReference snapshotReference1 = new MdsSnapshotReference(
                handle, 1, Instant.now().minus(Duration.standardDays(100)), Option.empty(), uid);

        MdsSnapshotReference snapshotReference2 = new MdsSnapshotReference(
                handle, 2, Instant.now(), Option.empty(), uid);

        mdsSnapshotReferenceJdbcDao.insert(snapshotReference1);
        mdsSnapshotReferenceJdbcDao.insert(snapshotReference2);

        mdsSnapshotReferenceManager.deleteOldSnapshotReferences();

        Assert.isEmpty(mdsSnapshotReferenceJdbcDao.find(uid, handle, 1));
        Assert.notEmpty(mdsSnapshotReferenceJdbcDao.find(uid, handle, 2));
    }

    @Test
    public void deleteOldSnapshotReferences_OldSnapshotWithMdsKey_DeletesDataFromMds() {
        String handle = Random2.R.nextAlnum(30);
        EllipticsUploadState state = ellipticsHelper.upload(Random2.R.nextString(10), new byte[] {1, 2, 3, 4});

        try{
            MdsSnapshotReference snapshotReference = new MdsSnapshotReference(
                    handle, 1, Instant.now().minus(Duration.standardDays(100)), Option.of(state.getKey()), uid);

            mdsSnapshotReferenceJdbcDao.insert(snapshotReference);
            mdsSnapshotReferenceManager.deleteOldSnapshotReferences();

            Option<MdsSnapshotReference> refO = mdsSnapshotReferenceJdbcDao.find(uid, handle, 1);
            Assert.isEmpty(refO);

            Assert.assertThrows(() -> ellipticsHelper.download(state.getKey()), EllipticsFileNotFoundException.class);
        } finally {
            // swallow EllipticsFileNotFoundException when key is missing
            ellipticsHelper.delete(state.getKey());
        }
    }

    @Test
    public void saveAndReadPartitionedSnapshot() {
        UserDatabaseSpec databaseSpec = new UserDatabaseSpec(uid,
                DatabaseRef.cons(Option.of("smartcache"), "photoslice"));

        dataApiManager.deleteDatabaseIfExists(databaseSpec, DatabaseDeletionMode.REMOVE_COMPLETELY);
        Database db = dataApiManager.getOrCreateDatabase(databaseSpec);

        ListF<RecordChange> changes = Cf.range(0, 102).map(i -> RecordChange.set(
                 "collection_" + i / 4, "record_" + i,
                Cf.map(
                        "field_1", DataField.string(Random2.R.nextAlnum(10)),
                        "field_2", DataField.integer(Random2.R.nextLong())
                )
        ));

        dataApiManager.applyDelta(db, RevisionCheckMode.WHOLE_DATABASE, new Delta(changes));

        db = dataApiManager.getAndInitCopyOnWrite(databaseSpec).get();

        mdsSnapshotReferenceManager.saveSnapshotToMdsIfNecessary(uid, db,
                () -> dataApiManager.getSnapshot(databaseSpec));

        Option<MdsSnapshotReference> referenceO =
                mdsSnapshotReferenceJdbcDao.find(uid, db.dbHandle.handle, db.rev);

        Assert.isTrue(referenceO.isPresent());
        //Get all data
        Snapshot s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT).get();
        Assert.sizeIs(102, s.records());

        //Get first collection
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionId("collection_0")).get();
        Assert.sizeIs(4, s.records());

        //Get second collection
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionId("collection_1")).get();
        Assert.sizeIs(4, s.records());

        //Get third collection (another mds key)
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionId("collection_2")).get();
        Assert.sizeIs(4, s.records());

        //Get several collections from two mds keys
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionIds(Cf.list("collection_1", "collection_3"))).get();
        Assert.sizeIs(8, s.records());

        //Get last collection from two mds keys
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionIds(Cf.list("collection_" + (101 / 4)))).get();
        Assert.sizeIs(2, s.records());

        mdsSnapshotReferenceManager.deleteStoredSnapshot(referenceO.get().mdsKey.get());
    }

    @Test
    public void saveAndReadPartitionedSnapshotWithBigCollection() {
        UserDatabaseSpec databaseSpec = new UserDatabaseSpec(uid,
                DatabaseRef.cons(Option.of("smartcache"), "photoslice"));

        dataApiManager.deleteDatabaseIfExists(databaseSpec, DatabaseDeletionMode.REMOVE_COMPLETELY);
        Database db = dataApiManager.getOrCreateDatabase(databaseSpec);

        ListF<RecordChange> changes = Cf.range(0, 8).map(i -> RecordChange.set(
                "collection_small", "record_" + i,
                Cf.map(
                        "field_1", DataField.string(Random2.R.nextAlnum(10)),
                        "field_2", DataField.integer(Random2.R.nextLong())
                )
        ));

        changes = changes.plus(Cf.range(0, 1000).map(i -> RecordChange.set(
                "collection_big" , "record_" + i,
                Cf.map(
                        "field_1", DataField.string(Random2.R.nextAlnum(10)),
                        "field_2", DataField.integer(Random2.R.nextLong())
                )
        )));

        changes = changes.plus(Cf.range(0, 4).map(i -> RecordChange.set(
                "collection_small_1" , "record_" + i,
                Cf.map(
                        "field_1", DataField.string(Random2.R.nextAlnum(10)),
                        "field_2", DataField.integer(Random2.R.nextLong())
                )
        )));

        changes = changes.plus(Cf.range(0, 4).map(i -> RecordChange.set(
                "collection_small_2" , "record_" + i,
                Cf.map(
                        "field_1", DataField.string(Random2.R.nextAlnum(10)),
                        "field_2", DataField.integer(Random2.R.nextLong())
                )
        )));

        changes = changes.plus(Cf.range(0, 4).map(i -> RecordChange.set(
                "collection_small_3" , "record_" + i,
                Cf.map(
                        "field_1", DataField.string(Random2.R.nextAlnum(10)),
                        "field_2", DataField.integer(Random2.R.nextLong())
                )
        )));

        dataApiManager.applyDelta(db, RevisionCheckMode.WHOLE_DATABASE, new Delta(changes));

        db = dataApiManager.getAndInitCopyOnWrite(databaseSpec).get();

        mdsSnapshotReferenceManager.saveSnapshotToMdsIfNecessary(uid, db,
                () -> dataApiManager.getSnapshot(databaseSpec));

        Option<MdsSnapshotReference> referenceO =
                mdsSnapshotReferenceJdbcDao.find(uid, db.dbHandle.handle, db.rev);

        Assert.isTrue(referenceO.isPresent());
        //Get small
        Snapshot s;

        //Get first small collection
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionId("collection_small")).get();
        Assert.sizeIs(8, s.records());

        //Get big collection
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionId("collection_big")).get();
        Assert.sizeIs(1000, s.records());

        //Get second small collection
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionId("collection_small_1")).get();
        Assert.sizeIs(4, s.records());

        //Get third small collection
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionId("collection_small_2")).get();
        Assert.sizeIs(4, s.records());

        //Get fourth small collection
        s = mdsSnapshotReferenceManager.getRevisionSnapshotFromMds(uid, db.rev, db.dbHandle,
                RecordsFilter.DEFAULT.withCollectionId("collection_small_3")).get();
        Assert.sizeIs(4, s.records());

        mdsSnapshotReferenceManager.deleteStoredSnapshot(referenceO.get().mdsKey.get());
    }
}
