package ru.yandex.chemodan.app.dataapi.core.ratelimiter.chunk.auto;

import java.util.concurrent.atomic.AtomicReference;

import lombok.AllArgsConstructor;
import org.junit.Test;

import ru.yandex.bolts.collection.Try;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.MigrationSupport;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.chemodan.ratelimiter.chunk.auto.AutoRwRateLimiter;
import ru.yandex.chemodan.ratelimiter.chunk.auto.HostKey;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MasterSlave;
import ru.yandex.chemodan.ratelimiter.chunk.auto.RateLimiterReference;
import ru.yandex.chemodan.ratelimiter.chunk.auto.ReadWrite;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
public class AutoRwContextAwareRateLimiterTest {

    private AtomicReference<HostKey> hostKeyHolder;

    @Test
    public void chooseWriteLimiter() {
        hostKeyHolder = new AtomicReference<>(new HostKey(0, MasterSlave.MASTER));

        ChunkRateLimiter limiter = new RateLimiterReference(new MockLimiterAutoContextAwareRwRateLimiter(), ReadWrite.READ);
        DataApiUserId uid = DataApiUserId.parse("1");

        MigrationSupport.doWithShardPolicy(
                uid, 1, MasterSlavePolicy.R_M, () -> limiter.acquirePermitAndExecuteV(size -> {}));
        Assert.equals(new HostKey(1, MasterSlave.MASTER), hostKeyHolder.get());

        MigrationSupport.doWithShardPolicy(
                uid, 2, MasterSlavePolicy.R_S, () -> limiter.acquirePermitAndExecuteV(size -> {}));
        Assert.equals(new HostKey(2, MasterSlave.SLAVE), hostKeyHolder.get());

        Assert.failure(Try.tryCatchException(() -> limiter.acquirePermitAndExecute(size -> 1.0)),
                IllegalStateException.class);
    }

    @AllArgsConstructor
    private class MockInnerLimiter implements ChunkRateLimiter {
        private final HostKey hostKey;

        @Override
        public int getDefaultChunkSize() {
            return 0;
        }

        @Override
        public <T> T acquirePermitAndExecute(int chunkSize, Function<Integer, T> action) {
            hostKeyHolder.set(hostKey);
            return action.apply(chunkSize);
        }
    }

    @AllArgsConstructor
    private class MockAutoRwLimiter implements AutoRwRateLimiter {
        ChunkRateLimiter innerLimiter;

        @Override
        public ChunkRateLimiter getReadRateLimiter() {
            return innerLimiter;
        }

        @Override
        public ChunkRateLimiter getWriteRateLimiter() {
            return innerLimiter;
        }

        @Override
        public void maintain() {
        }

        @Override
        public void updateMetrics() {

        }

        @Override
        public boolean hasAwaitingRequests() {
            return false;
        }

        @Override
        public boolean hasAvailableHosts() {
            return true;
        }
    }

    private class MockLimiterAutoContextAwareRwRateLimiter extends AutoContextAwareRwRateLimiter {
        public MockLimiterAutoContextAwareRwRateLimiter()
        {
            super(null, null, null, null, null, null, null, null, null,
                    null, null, null, null, null, null, null, null, null);
        }

        @Override
        protected AutoRwRateLimiter limiterConstructor(HostKey key) {
            return new MockAutoRwLimiter(new MockInnerLimiter(key));
        }
    }
}
