package ru.yandex.chemodan.app.dataapi.core.xiva;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.api.DatabaseChangedEventAsyncHandler;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.datasources.yamoney.YaMoneyDatabaseRef;
import ru.yandex.chemodan.app.dataapi.utils.xiva.XivaClientMultiplexed;
import ru.yandex.chemodan.xiva.XivaEvent;
import ru.yandex.chemodan.xiva.XivaPushBody;
import ru.yandex.chemodan.xiva.XivaPushBodyRepack;
import ru.yandex.chemodan.xiva.XivaPushService;
import ru.yandex.chemodan.xiva.XivaSendResponse;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.MarshallerContext;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
public class DataApiXivaPushSender implements DatabaseChangedEventAsyncHandler {
    private static final Logger logger = LoggerFactory.getLogger(DataApiXivaPushSender.class);
    private static final AppDatabaseRef PHOTOSLICE_DB_REF = new AppDatabaseRef("smartcache", "photoslice");

    private static final SetF<DatabaseRef> EXCLUDE_DB_REFS = Cf.<DatabaseRef>set()
            .plus(PHOTOSLICE_DB_REF)
            .plus(Cf.x(YaMoneyDatabaseRef.values())
                    .map(YaMoneyDatabaseRef::dbRef)
            );
    public static final String DATASYNC_DATABASE_CHANGED_EVENT = "datasync_database_changed";

    private final DynamicProperty<ListF<String>> collapsableDatabases = DynamicProperty.cons(
            "collapsable-database-keys", Cf.list());

    private final XivaClientMultiplexed.SingleToken xivaPushClient;

    private final ExternalDatabasesRegistry externalDatabasesRegistry;

    private final AppSettingsRegistry appSettingsRegistry;

    public DataApiXivaPushSender(
            XivaClientMultiplexed.SingleToken xivaPushClient,
            ExternalDatabasesRegistry externalDatabasesRegistry,
            AppSettingsRegistry appSettingsRegistry)
    {
        this.externalDatabasesRegistry = externalDatabasesRegistry;
        this.xivaPushClient = xivaPushClient;
        this.appSettingsRegistry = appSettingsRegistry;
    }

    @Override
    public void databaseChanged(DatabaseChange databaseChange) {
        sendPushesAboutChangesInDataSync(databaseChange.patchedDatabase());
    }

    private void sendPushesAboutChangesInDataSync(Database database) {
        if (!isPushSendingNeeded(database)) {
            return;
        }

        for (String databaseId : getAllIdsOfSpecificDatabase(database)) {
            doSendPush(database.uid, database.dbContext().consDbRef(databaseId), database.rev);
        }
    }

    public String doSendPush(DataApiUserId uid, DatabaseRef dbRef, long rev) {
        Context context = dbRef.hasAppContext() ? Context.APP : Context.USER;
        String contextString = context.toString().toLowerCase();
        String databaseId = dbRef.databaseId();
        String aliasId = (uid.isPublic() ? ".pub." + dbRef.dbAppId() + "@" : "") + databaseId;

        ListF<String> tags = (context == Context.APP && !uid.isPublic())
                ? Cf.list(databaseId, contextString + ":" + databaseId)
                : Cf.list(contextString + ":" + aliasId);

        Option<String> collapseKey = collapsableDatabases.get().find(databaseId::equals);

        XivaEvent xivaEvent = new XivaEvent(DataApiXivaUtils.toXivaRecipient(uid), DATASYNC_DATABASE_CHANGED_EVENT)
                .addTags(tags)
                .withBody(new DatabaseChangedEventV2ForXiva(
                        DATASYNC_DATABASE_CHANGED_EVENT, rev, aliasId, contextString, collapseKey));

        XivaSendResponse response = xivaPushClient.byUserId(uid).send(xivaEvent);
        String transitId = response.headers.transitId.getOrNull();
        logger.info("Sent {} event, transit id: {}", DATASYNC_DATABASE_CHANGED_EVENT, transitId);

        return transitId;
    }

    private boolean isPushSendingNeeded(Database db) {
        return DataApiXivaUtils.isUidSupportedForSending(db.uid)
                && !EXCLUDE_DB_REFS.containsTs(db.dbRef())
                && !appSettingsRegistry.getDatabaseSettings(db).isNoXiva();
    }

    private ListF<String> getAllIdsOfSpecificDatabase(Database database) {
        return externalDatabasesRegistry.getExternalIdsByInternal(database.databaseId())
                .plus1(database.databaseId());
    }

    @BenderBindAllFields
    public static class DatabaseChangedEventV2ForXiva implements XivaPushBody {

        private final static BenderMapper benderMapper = DataApiBenderUtils.mapper();

        public final long revision;
        @BenderPart(name="t", strictName = true)
        public final String eventType;
        @BenderPart(name = "database_id", strictName = true)
        public final String databaseId;
        public final String context;

        @BenderIgnore
        private final Option<String> collapseKey;

        public DatabaseChangedEventV2ForXiva(String eventType, long revision, String databaseId, String context,
                Option<String> collapseKey) {
            this.eventType = eventType;
            this.revision = revision;
            this.databaseId = databaseId;
            this.context = context;
            this.collapseKey = collapseKey;
        }

        @Override
        public void writePushBodyAsJson(BenderJsonWriter writer, MarshallerContext context) {
            benderMapper.serializeJson(this, writer, context);
        }

        @Override
        public XivaPushBodyRepack getRepack() {
            return new XivaPushBodyRepack()
                    .withServiceRepack(XivaPushService.GCM,
                            repack -> collapseKey.ifPresent(key -> repack.addStandardField("collapse_key", key))
                    )
                    .withServiceRepack(XivaPushService.APNS)
                    .includeSpecialAndAllPayloadFieldsToAllServices();
        }
    }

    private enum Context {
        APP,
        USER
    }
}
