package ru.yandex.chemodan.app.dataapi.core.xiva;

import java.util.List;

import com.xebialabs.restito.semantics.Call;
import com.xebialabs.restito.server.StubServer;
import org.glassfish.grizzly.http.Method;
import org.glassfish.grizzly.http.util.HttpStatus;
import org.joda.time.Instant;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseMeta;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPublicUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.dataapi.test.TestConstants;
import ru.yandex.chemodan.app.dataapi.utils.xiva.XivaClientMultiplexed;
import ru.yandex.chemodan.util.JsonAssert;
import ru.yandex.chemodan.util.test.StubServerUtils;
import ru.yandex.chemodan.xiva.BasicXivaClient;
import ru.yandex.chemodan.xiva.DiskXivaServices;
import ru.yandex.chemodan.xiva.XivaClient;
import ru.yandex.chemodan.xiva.XivaUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.http.HttpHeaderNames;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.test.Assert;

import static com.xebialabs.restito.builder.stub.StubHttp.whenHttp;
import static com.xebialabs.restito.semantics.Action.status;
import static com.xebialabs.restito.semantics.Condition.method;

/**
 * @author metal
 */
@ActivateDataApiEmbeddedPg
public class DataApiXivaPushSenderTest extends DataApiTestSupport {
    @Autowired
    private ExternalDatabasesRegistry externalDatabasesRegistry;
    @Autowired
    private AppSettingsRegistry appSettingsRegistry;

    @Value("${xiva.stoken}")
    private String xivaStoken;

    @Test
    public void sendPushToXivaAfterDatabaseChanges() {
        StubServerUtils.withStubServer((port, stubServer) -> {
            DataApiPassportUserId uid = new DataApiPassportUserId(101);

            List<Call> calls = getCallsAfterDatabaseChanges(port, stubServer, uid, 59);
            Assert.equals(1, calls.size());

            String databaseId = TestConstants.CLIENT_DB_REF.databaseId();
            String expectedBodyStart = "{" +
                    "\"payload\":{" +
                        "\"revision\":59," +
                        "\"t\":\"datasync_database_changed\"," +
                        "\"database_id\":\"" + databaseId + "\"," +
                        "\"context\":\"app\"" +
                    "}," +
                    "\"tags\":[\"" + databaseId + "\",\"app_" + databaseId + "\"]," +
                    "\"repack\":{";
            String expectedApnsPart =
                        "\"apns\":{" +
                            "\"repack_payload\":[" +
                                "\"t\"," +
                                "\"revision\"," +
                                "\"database_id\"," +
                                "\"context\"," +
                                "{\"r\":\"::xiva::push_token\"}," +
                                "{\"transit_id\":\"::xiva::transit_id\"}" +
                            "]" +
                        "}";
            String expectedOtherPart =
                        "\"other\":{" +
                            "\"repack_payload\":[" +
                                "\"t\"," +
                                "\"revision\"," +
                                "\"database_id\"," +
                                "\"context\"," +
                                "{\"r\":\"::xiva::push_token\"}," +
                                "{\"transit_id\":\"::xiva::transit_id\"}" +
                            "]" +
                        "}";

            JsonAssert.equalsLenient(
                    DataApiXivaPushSenderTest.class,
                    "sendPushToXivaAfterDatabaseChanges.json",
                    calls.get(0).getPostBody()
            );

            assertContainsParameter(calls.get(0), "uid", "101");
            assertContainsHeader(calls.get(0), HttpHeaderNames.AUTHORIZATION, "Xiva " + xivaStoken);
        });
    }

    @Test
    public void sendPushToXivaAfterPublicDatabaseChanges() {
        StubServerUtils.withStubServer((port, stubServer) -> {
            DataApiUserId uid = new DataApiPublicUserId();

            List<Call> calls = getCallsAfterDatabaseChanges(port, stubServer, uid, 60);
            Assert.equals(1, calls.size());

            String databaseId = TestConstants.CLIENT_DB_REF.databaseId();
            String appId = TestConstants.CLIENT_DB_REF.dbAppId();

            String aliasId = ".pub." + appId + "@" + databaseId;
            String tag = XivaUtils.formatTag("app:" + aliasId);

            String expectedDbId = "\"database_id\":\"" + aliasId + "\",";
            String expectedTags = "\"tags\":[\"" + tag + "\"]";

            Assert.assertContains(calls.get(0).getPostBody(), expectedDbId);
            Assert.assertContains(calls.get(0).getPostBody(), expectedTags);
            Assert.equals("/beta/wild/send", calls.get(0).getUri());
            Assert.isFalse(calls.get(0).getParameters().containsKey("uid"));
            assertContainsHeader(calls.get(0), HttpHeaderNames.AUTHORIZATION,
                    "Xiva " + xivaStoken + "_wild");
        });
    }

    private List<Call> getCallsAfterDatabaseChanges(int port, StubServer stubServer, DataApiUserId uid, int revision) {
        whenHttp(stubServer).match(method(Method.POST)).then(status(HttpStatus.OK_200));

        BasicXivaClient basicXivaClient = new BasicXivaClient("http://localhost:" + port,
                ApacheHttpClientUtils.Builder.create()
                        .singleThreaded()
                        .withMaxConnections(1)
                        .withRequestRetry(0, false)
                        .build()
        );
        MapF<String, String> tokens = Cf.map(
                DiskXivaServices.DATASYNC, xivaStoken,
                DiskXivaServices.DATASYNC_BROADCAST, xivaStoken + "_wild"
        );

        XivaClient xivaClient = new XivaClient(basicXivaClient, tokens);
        DataApiXivaPushSender dataApiXivaPushSender = new DataApiXivaPushSender(
                new XivaClientMultiplexed(xivaClient, xivaClient)
                        .toSingleTokenClient(DiskXivaServices.DATASYNC, DiskXivaServices.DATASYNC_BROADCAST),
                externalDatabasesRegistry, appSettingsRegistry);
        Database database = new Database(
                uid,
                TestConstants.CLIENT_DB_REF
                        .consHandle("handle"),
                revision,
                new DatabaseMeta(Instant.now(), Instant.now(), DataSize.fromBytes(1), 0));

        dataApiXivaPushSender.databaseChanged(DatabaseChange.empty(database));

        return stubServer.getCalls();
    }

    @SuppressWarnings("SameParameterValue")
    private static void assertContainsParameter(Call call, String expectedParameter, String expectedValue) {
        String[] actualValues = call.getParameters().get(expectedParameter);
        Assert.notNull(actualValues, "Expected parameter " + expectedParameter + " is missing");
        Assert.equals(expectedValue, actualValues[0]);
    }

    @SuppressWarnings("SameParameterValue")
    private static void assertContainsHeader(Call call, String expectedHeader, String expectedValue) {
        Assert.equals(expectedValue, call.getHeaders().get(expectedHeader.toLowerCase()));
    }
}
