package ru.yandex.chemodan.app.dataapi.core.xiva;

import java.util.LinkedHashMap;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiYaTeamUserId;
import ru.yandex.chemodan.util.test.TestUser;

/**
 * @author metal
 */
public class XivaUrlHelperTest {
    private final String XIVA_URL_BASE = "test_xivaUrl.yandex";
    private final String XIVA_URL = XIVA_URL_BASE + ".ru";
    private final String XIVA_YATEAM_URL_BASE = "test_xivaUrl.yandex-team";
    private final String XIVA_YATEAM_URL = XIVA_YATEAM_URL_BASE + ".ru";

    private final DataApiUserId UID = new DataApiPassportUserId(TestUser.uid);
    private final DataApiUserId YT_UID = new DataApiYaTeamUserId(1120000000000001L);
    private final String XIVA_STOKEN = "xiva_stoken";
    private final String XIVA_CTOKEN = "xiva_ctoken";
    private final String OAUTH_TOKEN = "user_oauth_token";
    private final String CLIENT = "user_client";
    private final String SESSION = "user_session";
    private final String DEVICE_ID = "user_device_id";
    private final String PLATFORM = "user_platform";
    private final String PLATFORM_PUSH_TOKEN = "user_platform_push_token";
    private final String CALLBACK = "user_callback";

    private XivaUrlHelper xivaUrlHelper = new XivaUrlHelper(XIVA_URL, XIVA_YATEAM_URL);
    private MapF<String, ListF<String>> servicesWithTags;
    private ListF<String> tags = Cf.list("tag_1@test", "tag_2", "tag_3", "tag_4");

    @Before
    public void initialize() {
        servicesWithTags = Cf.x(new LinkedHashMap<>());
        servicesWithTags.put("service_1", Cf.list("tag_1", "tag_21*tag_22", "tag_3", "tag_4"));
        servicesWithTags.put("service_2", Cf.list("tag_5@test", "tag_6"));
        servicesWithTags.put("service_3", Cf.list());
    }

    @Test
    public void getStreamSubscriptionUrl() {
        String rightUrlSuffix = "/v1/subscribe" +
                "?oauth_token=user_oauth_token" +
                "&uid=201515217" +
                "&client=user_client" +
                "&session=user_session" +
                "&service=service_1%3Atag_1%2Btag_21*tag_22%2Btag_3%2Btag_4%2Cservice_2%3Atag_5_test%2Btag_6%2Cservice_3";

        checkStreamSubscriptionUrl(XIVA_URL_BASE, rightUrlSuffix, UID);
    }

    @Test
    public void getStreamSubscriptionUrlForYaTeam() {
        String rightUrlSuffix = "/v1/subscribe" +
                "?oauth_token=user_oauth_token" +
                "&uid=1120000000000001" +
                "&client=user_client" +
                "&session=user_session" +
                "&service=service_1%3Atag_1%2Btag_21*tag_22%2Btag_3%2Btag_4%2Cservice_2%3Atag_5_test%2Btag_6%2Cservice_3";

        checkStreamSubscriptionUrl(XIVA_YATEAM_URL_BASE, rightUrlSuffix, YT_UID);
    }

    private void checkStreamSubscriptionUrl(String xivaUrlBase, String rightUrlSuffix, DataApiUserId uid) {
        XivaUrlGeneratingTestUtils.assertUrlEquals(xivaUrlBase + ".ru" + rightUrlSuffix,
                xivaUrlHelper.getStreamSubscriptionUrl(Option.of(OAUTH_TOKEN), uid, CLIENT, SESSION, servicesWithTags,
                        Option.empty()));
        checkStreamSubscriptionUrlWithTld(xivaUrlBase, rightUrlSuffix, uid, "ru");
        checkStreamSubscriptionUrlWithTld(xivaUrlBase, rightUrlSuffix, uid, "ua");
        checkStreamSubscriptionUrlWithTld(xivaUrlBase, rightUrlSuffix, uid, "com.tr");
    }

    private void checkStreamSubscriptionUrlWithTld(
            String xivaUrlBase, String rightUrlSuffix, DataApiUserId uid, String tld)
    {
        String tldWithDot = "." + tld;

        XivaUrlGeneratingTestUtils.assertUrlEquals(xivaUrlBase + tldWithDot + rightUrlSuffix,
                xivaUrlHelper.getStreamSubscriptionUrl(Option.of(OAUTH_TOKEN), uid, CLIENT, SESSION,
                        servicesWithTags, Option.of(tld)));
        XivaUrlGeneratingTestUtils.assertUrlEquals(xivaUrlBase + tldWithDot + rightUrlSuffix,
                xivaUrlHelper.getStreamSubscriptionUrl(Option.of(OAUTH_TOKEN), uid, CLIENT, SESSION,
                        servicesWithTags, Option.of(tldWithDot)));
    }

    @Test
    public void getApplicationSubscriptionUrl() {
        XivaUrlGeneratingTestUtils.assertUrlEquals(XIVA_URL + "/v1/subscribe/app" +
                        "?client=user_client" +
                        "&deviceId=user_device_id" +
                        "&platform=user_platform" +
                        "&platform_push_token=user_platform_push_token" +
                        "&session=user_session" +
                        "&service=service_1%3Atag_1%2Btag_21*tag_22%2Btag_3%2Btag_4%2Cservice_2%3Atag_5_test%2Btag_6%2Cservice_3",
                xivaUrlHelper.getApplicationSubscriptionUrl(Option.empty(), Option.of(CLIENT), SESSION,
                        Option.of(DEVICE_ID), PLATFORM, PLATFORM_PUSH_TOKEN, servicesWithTags));
    }

    @Test
    public void getCallbackSubscriptionUrl() {
        XivaUrlGeneratingTestUtils.assertUrlEquals(XIVA_URL + "/v1/subscribe/url" +
                        "?uid=201515217" +
                        "&ctoken=xiva_ctoken" +
                        "&callback=user_callback" +
                        "&session=user_session" +
                        "&service=service_1%3Atag_1%2Btag_21*tag_22%2Btag_3%2Btag_4%2Cservice_2%3Atag_5_test%2Btag_6%2Cservice_3",
                xivaUrlHelper.getCallbackSubscriptionUrl(UID, XIVA_CTOKEN, CALLBACK,
                        SESSION, servicesWithTags));
    }

    @Test
    public void getCallbackSubscriptionUrlForYaTeam() {
        XivaUrlGeneratingTestUtils.assertUrlEquals(XIVA_YATEAM_URL + "/v1/subscribe/url" +
                        "?uid=1120000000000001" +
                        "&ctoken=xiva_ctoken" +
                        "&callback=user_callback" +
                        "&session=user_session" +
                        "&service=service_1%3Atag_1%2Btag_21*tag_22%2Btag_3%2Btag_4%2Cservice_2%3Atag_5_test%2Btag_6%2Cservice_3",
                xivaUrlHelper.getCallbackSubscriptionUrl(YT_UID, XIVA_CTOKEN, CALLBACK,
                        SESSION, servicesWithTags));
    }
}
