package ru.yandex.chemodan.app.dataapi.maintenance;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseAppContext;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.commune.db.shard2.ShardManager2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author akirakozov
 */
public class MapPointsHistoryCleaner extends DelayingWorkerServiceBeanSupport {
    private static final Logger logger = LoggerFactory.getLogger(MapPointsHistoryCleaner.class);

    private static final DatabaseAppContext DB_CTX = new DatabaseAppContext("maps_common");
    private static final AppDatabaseRef DB_REF = DB_CTX.consDbRef("ymapspointshistory1");

    private static final CollectionRef POINTS_HISTORY_COL_REF = DB_REF.consColRef("pointshistory");

    @Value("${dataapi.map_history.cleaner.enabled}")
    private boolean enabled;
    @Value("${dataapi.map_history.cleaner.recordsSearchLimit}")
    private int recordsSearchLimit;
    @Value("${dataapi.map_history.cleaner.recordsEnough}")
    private int recordsEnough;

    @Autowired
    private ShardManager2 dataShardManager;
    @Autowired
    private DataApiManager dataApiManager;

    @Override
    @Value("${dataapi.map_history.cleaner.delay}")
    public void setDelay(Duration delay) {
        super.setDelay(delay);
    }

    @Override
    protected void execute() throws Exception {
        if (!enabled) {
            logger.info("Cleaning is disabled");
        } else {
            logger.info("Cleaning is enabled");
            cleanAll();
        }
    }

    void cleanAll() {
        Instant start = Instant.now();
        final int[] count = {0};
        JdbcTemplate3 template = new JdbcTemplate3(dataShardManager.getShard(1).getDataSource());
        Cf.range(0, 128).iterator().flatMap(p -> {
            String part = StringUtils.leftPad(Integer.toString(p), 3, '0');

            logger.info("Processing partition " + part);

            return template.queryForList(
                    "SELECT user_id FROM databases_" + part + " db WHERE db.dbId = '"
                        + DB_REF.databaseId() + "' AND db.app = '"
                        + DB_CTX.appName() + "' AND records_count > " + recordsSearchLimit,
                    String.class
                ).iterator();
        }).forEachRemaining(u -> {
            cleanup(DataApiUserId.parse(u));

            if (++count[0] % 100 == 0) {
                logger.info("" + count[0] + " processed in " + TimeUtils.secondsStringToNow(start));
            }
        });
        logger.info("Processed {} in {}", count[0], TimeUtils.secondsStringToNow(start));
    }

    private void cleanup(DataApiUserId uid) {
        try {
            Snapshot snapshot = dataApiManager.getSnapshot(
                    new UserDatabaseSpec(uid, POINTS_HISTORY_COL_REF.dbRef()),
                    RecordsFilter.DEFAULT
                            .withColRef(POINTS_HISTORY_COL_REF)
            );

            logger.info("Common records count: " + snapshot.recordCount());
            ListF<DataRecord> recordsFiltered =
                    snapshot.records().toList().filter(o -> o.getData().containsKeyTs("last_used"));
            logger.info("Common records with last_used field: " + recordsFiltered.size());

            ListF<DataRecord> records = recordsFiltered
                    .sortedBy(o -> o.getData().getTs("last_used").timestampValue())
                    .rdrop(recordsEnough);

            records = records.plus(snapshot.records().filter(o -> !o.getData().containsKeyTs("last_used")));

            if (records.isNotEmpty()) {
                Delta delta = new Delta(records.map(rec -> RecordChange.delete(rec.id)));

                dataApiManager.applyDelta(new UserDatabaseSpec(uid, DB_REF), snapshot.database.rev,
                        RevisionCheckMode.PER_RECORD, delta);

                logger.info("User " + uid + " cleaned up, records deleted: " + records.size());
            } else {
                logger.info("User " + uid + " skipped since nothing to delete");
            }
        } catch (Exception e) {
            logger.info("User " + uid + " failed:" + e);
        }
    }
}
