package ru.yandex.chemodan.app.dataapi.support;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.ConvertingDataColumn;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataColumn;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;

/**
 * @author dbrylev
 */
public class RecordField<T> {

    public final String name;
    public final Function<DataField, T> fromF;
    public final Function<T, DataField> toF;

    private final Option<ConvertingDataColumn<T>> column;

    public RecordField(
            String name, Function<DataField, T> fromF, Function<T, DataField> toF,
            Function<String, DataColumn<T>> columnF)
    {
        this(name, fromF, toF, columnF, t -> t);
    }

    public <C> RecordField(
            String name, Function<DataField, T> fromF,  Function<T, DataField> toF,
            Function<String, DataColumn<C>> columnF, Function<T, C> converterF)
    {
        this(name, fromF, toF, Option.of(new ConvertingDataColumn<>(columnF.apply(name), converterF)));
    }

    public RecordField(String name,
            Function<DataField, T> fromF, Function<T, DataField> toF, Option<ConvertingDataColumn<T>> column)
    {
        this.name = name;
        this.fromF = fromF;
        this.toF = toF;
        this.column = column;
    }

    public T get(DataRecord rec) {
        return get(rec.getData());
    }

    public T get(DataFieldsSource src) {
        return get(src.toData());
    }

    public T get(MapF<String, DataField> data) {
        return fromF.apply(data.getOrThrow(name));
    }

    public Option<T> getO(DataRecord rec) {
        return rec.getData().getO(name).map(fromF);
    }

    public Option<T> getO(DataFieldsSource src) {
        return getO(src.toData());
    }

    public Option<T> getO(MapF<String, DataField> data) {
        return data.getO(name).map(fromF);
    }

    public Tuple2<String, DataField> toData(T value) {
        return Tuple2.tuple(name, toF.apply(value));
    }

    public FieldChange changedTo(T value) {
        return FieldChange.put(name, toF.apply(value));
    }

    public ConvertingDataColumn<T> column() {
        return column.getOrThrow("Column is not defined");
    }

    public static RecordField<Boolean> bool(String name) {
        return new RecordField<>(name, DataField::booleanValue, DataField::bool, DataColumn::bool);
    }

    public static RecordField<String> string(String name) {
        return new RecordField<>(name, DataField::stringValue, DataField::string, DataColumn::string);
    }

    public static RecordField<Long> longNumber(String name) {
        return new RecordField<>(name, DataField::integerValue, DataField::integer, DataColumn::integer);
    }

    public static RecordField<Integer> intNumber(String name) {
        return new RecordField<>(name,
                Cf.Long.toIntegerF().compose(DataField::integerValue),
                Cf.Integer.toLongF().andThen(DataField::integer),
                DataColumn::integer, Cf.Integer.toLongF());
    }

    public static RecordField<Double> decimal(String name) {
        return new RecordField<>(name, DataField::decimalValue, DataField::decimal, DataColumn::decimal);
    }

    public static RecordField<Instant> instant(String name) {
        return new RecordField<>(name, DataField::timestampValue, DataField::timestamp, DataColumn::timestamp);
    }

    public static <T extends Enum<T> & StringEnum> RecordField<T> stringEnum(String name, Class<T> enumClass) {
        StringEnumResolver<T> resolver = StringEnumResolver.r(enumClass);
        return new RecordField<>(
                name, d -> resolver.fromValue(d.stringValue()), v -> DataField.string(v.value()),
                DataColumn::string, v -> v.value());
    }

    public static <T> RecordField<ListF<T>> list(RecordField<T> base) {
        return new RecordField<>(
                base.name, d -> d.listValue().map(base.fromF), v -> DataField.list(v.map(base.toF)), Option.empty());
    }
}
