package ru.yandex.chemodan.app.dataapi.utils;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.springframework.core.Ordered;

import ru.yandex.chemodan.app.dataapi.core.DatabasesContextConfiguration;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;

/**
 * Executor for {@link PriorityRunnable }
 * @author vpronto
 */
public class PriorityExecutorService implements ExecutorService {

    private static final Logger logger = LoggerFactory.getLogger(DatabasesContextConfiguration.class);

    private final ThreadPoolExecutor asyncExecutorSlow;
    private final ThreadPoolExecutor asyncExecutorFast;

    public PriorityExecutorService(int corePoolSizeSlow, int corePoolSizeFast, int maximumPoolSize,
                                   long keepAliveTimeMinutes)
    {

        asyncExecutorSlow = new ThreadPoolExecutor(corePoolSizeSlow, maximumPoolSize,
                keepAliveTimeMinutes, TimeUnit.MINUTES,
                new ArrayBlockingQueue<>(500),
                new ThreadNameIndexThreadFactory("database-change-thread-slow"),
                (r, executor) -> logger.error("Discarded db change! {}", r));

        asyncExecutorFast = new ThreadPoolExecutor(corePoolSizeFast, maximumPoolSize,
                keepAliveTimeMinutes, TimeUnit.MINUTES,
                new ArrayBlockingQueue<>(50),
                new ThreadNameIndexThreadFactory("database-change-thread-fast"),
                new ThreadPoolExecutor.CallerRunsPolicy() {
                    public void rejectedExecution(Runnable r, ThreadPoolExecutor e) {
                        logger.warn("TP database-change-thread-fast can't process all tasks");
                        super.rejectedExecution(r, e);
                    }
                });

        asyncExecutorSlow.prestartAllCoreThreads();
        asyncExecutorFast.prestartAllCoreThreads();
    }

    private ExecutorService es(Runnable command) {

        if (command instanceof PriorityRunnable) {
            PriorityRunnable priorityRunnable = (PriorityRunnable) command;
            if (priorityRunnable.getOrder() == Ordered.HIGHEST_PRECEDENCE) {
                return asyncExecutorFast;
            }
        }
        return es();
    }

    private ExecutorService es() {
        return asyncExecutorSlow;
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit)
            throws InterruptedException {
        return es().awaitTermination(timeout, unit);
    }

    @Override
    public <T> List<Future<T>> invokeAll(
            Collection<? extends Callable<T>> tasks) throws InterruptedException {
        return es().invokeAll(tasks);
    }

    @Override
    public <T> List<Future<T>> invokeAll(
            Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
            throws InterruptedException {
        return es().invokeAll(tasks, timeout, unit);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
            throws InterruptedException, ExecutionException {
        return es().invokeAny(tasks);
    }

    @Override
    public <T> T invokeAny(
            Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
            throws InterruptedException, ExecutionException, TimeoutException {
        return es().invokeAny(tasks, timeout, unit);
    }

    @Override
    public boolean isShutdown() {
        return es().isShutdown();
    }

    @Override
    public boolean isTerminated() {
        return es().isTerminated();
    }

    @Override
    public void shutdown() {
        es().shutdown();
    }

    @Override
    public List<Runnable> shutdownNow() {
        return es().shutdownNow();
    }

    public <T> Future<T> submit(Callable<T> task) {
        return es().submit(task);
    }

    @Override
    public void execute(Runnable command) {
        es(command).execute(command);
    }

    @Override
    public Future<?> submit(Runnable task) {
        return es(task).submit(task);
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
        return es(task).submit(task, result);
    }
}
