package ru.yandex.chemodan.app.dataapi.utils.dataconversion;

import java.io.StringWriter;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonNode;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.ArrayTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.BooleanTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.ByteArrayTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.DateTimeTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.FlatObjectTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.IntegerTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.NumberTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.ObjectTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.StringTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.TimestampTypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.typeconverters.TypeConverter;
import ru.yandex.chemodan.app.dataapi.utils.serializers.datafieldmap.DataFieldNode;
import ru.yandex.chemodan.app.dataapi.utils.serializers.datafieldmap.DataFieldWriter;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.parse.JacksonJsonNodeWrapper;

/**
 * @author Denis Bakharev
 */
public class FormatConverter {
    private final NodeConversionSettings rootNodeSettings;
    public final String jsonSchema;

    private static final MapF<String, TypeConverter> typeConverters;

    static {
        ListF<TypeConverter> converters = Cf.list(
                new ArrayTypeConverter(),
                new ByteArrayTypeConverter(),
                new FlatObjectTypeConverter(),
                new IntegerTypeConverter(),
                new ObjectTypeConverter(),
                new BooleanTypeConverter(),
                new DateTimeTypeConverter(),
                new NumberTypeConverter(),
                new StringTypeConverter(),
                new DateTimeTypeConverter(),
                new TimestampTypeConverter());

        typeConverters = converters.toMapMappingToKey(TypeConverter::getConvertedTypeName);
    }

    public FormatConverter(String jsonSchema) {
        this.jsonSchema = jsonSchema;
        rootNodeSettings = ConversionSettingsLoader.loadFromJsonSchema(jsonSchema);
    }

    public MapF<String, DataField> toDataFields(String jsonData) {
        return toDataFields(JsonNodeUtils.getNode(jsonData));
    }

    public MapF<String, DataField> toDataFields(JsonNode rootNode) {
        DataFieldWriter writer = new DataFieldWriter();
        JsonToDataFieldConverter converter = new JsonToDataFieldConverter(typeConverters, writer);
        converter.convert(new ConverterContext<>(Option.empty(), rootNodeSettings, new JacksonJsonNodeWrapper(rootNode)));
        return writer.getResult();
    }

    public String toJson(MapF<String, DataField> map) {
        StringWriter stringWriter = new StringWriter();
        JsonFactory jsonFactory = new JsonFactory();
        try (JsonGenerator generator = jsonFactory.createGenerator(stringWriter)) {
            DataField rootNode = DataField.map(map);

            DataFieldToJsonConverter converter = new DataFieldToJsonConverter(typeConverters, generator);
            converter.convert(new ConverterContext<>(Option.empty(), rootNodeSettings, new DataFieldNode(rootNode)));

        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }

        return stringWriter.toString();
    }

    public Option<TypeConverter> getFieldConverter(String fieldName) {
        return rootNodeSettings.children.getO(fieldName).map(s -> typeConverters.getOrThrow(s.getActualType()));
    }
}
