package ru.yandex.chemodan.app.dataapi.utils.dataconversion;

import java.util.Base64;

import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.misc.test.Assert;

/**
 * @author Denis Bakharev
 */
public class FormatConverterTest extends DataConversionUnitTestBase {
    private final String jsonSchema =
            "{\n"
            + "  \"type\": \"object\",\n"
            + "  \"properties\": {\n"
            + "    \"a\": {\"type\": \"integer\"},\n"
            + "    \"/\": {\"type\": \"integer\"},\n"
            + "    \"~\": {\"type\": \"integer\"},\n"
            + "    \".\": {\"type\": \"integer\"}\n"
            + "  }\n"
            + "}";

    @Test
    public void simpleSchemaWithFullDataTest() {
        String jsonInput =
                "{ \"a\":1, \"/\":2, \"~\":3, \".\":4 }";

        MapF<String, DataField> map = doConvertCycle(jsonSchema, jsonInput);

        Assert.equals(4, map.size());
        assertContains(map, "a", DataField.integer(1));
        assertContains(map, "/", DataField.integer(2));
        assertContains(map, "~", DataField.integer(3));
        assertContains(map, ".", DataField.integer(4));
    }

    @Test
    public void simpleSchemaWithMinDataTest() {
        String jsonInput =
                "{ \"a\":1, \".\":4 }";

        MapF<String, DataField> map = doConvertCycle(jsonSchema, jsonInput);

        Assert.equals(2, map.size());
        assertContains(map, "a", DataField.integer(1));
        assertContains(map, ".", DataField.integer(4));
    }

    @Test
    public void complexSchemaWithFullDataTest() {
        String jsonSchema = readResource("ComplexJsonSchema.json");
        String inputJson = readResource("ComplexFullJsonData.json");

        MapF<String, DataField> root = doConvertCycle(jsonSchema, inputJson);

        Assert.equals(7, root.size());

        DataField objTree = root.getTs("objTree");

        assertContains(objTree.mapValue(), "innerObj", DataField.map(Cf.map("intProp", DataField.integer(1))));
        assertContains(objTree.mapValue(), "arrayOfInts", DataField.list(DataField.integer(2), DataField.integer(3)));

        ListF<DataField> list = objTree.mapValue().getTs("arrayOfObjects").listValue();
        assertContains(list.get(0).mapValue(), "objProp", DataField.map(Cf.map("intProp", DataField.integer(100))));
        assertContains(list.get(1).mapValue(), "objProp", DataField.map(Cf.map("intProp", DataField.integer(200))));

        assertContains(root, "numberProp", DataField.decimal(101.101));
        assertContains(root, "propBytes", DataField.bytes(Base64.getDecoder().decode("aGVsbG8=")));
        assertContains(root, "flatObj_a_/", DataField.integer(20));
        assertContains(root, "flatObj_a_prop1", DataField.integer(30));
        assertContains(root, "flatObj_a_prop2", DataField.integer(40));
        assertContains(root, "flatObj_b", DataField.integer(50));
    }

    @Test
    public void complexSchemaWithMinDataTest() {
        String jsonSchema = readResource("ComplexJsonSchema.json");
        String inputJson = readResource("ComplexMinJsonData.json");

        MapF<String, DataField> root = doConvertCycle(jsonSchema, inputJson);

        Assert.equals(2, root.size());
        DataField objTree = root.getTs("objTree");
        assertContains(objTree.mapValue(), "arrayOfObjects", DataField.list());
        assertContains(root, "flatObj_a_prop1", DataField.integer(30));
    }


    @Test
    public void convertWithNullField() {
        String json = "{\n" +
                "  \"type\": \"object\",\n" +
                "  \"properties\": {\n" +
                "    \"id\": {\n" +
                "      \"type\": \"string\"\n" +
                "    },\n" +
                "    \"created_at\": {\n" +
                "      \"type\": \"string\",\n" +
                "      \"dataapi-convert-type\": \"timestamp\"\n" +
                "    }" +
                "  },\n" +
                "  \"required\": [\n" +
                "    \"id\"\n" +
                "  ]\n" +
                "}";

        NodeConversionSettings settings = ConversionSettingsLoader.loadFromJsonSchema(json);
        FormatConverter converter = new FormatConverter(json);

        String recordJson = converter.toJson(Cf.map(
                "id", DataField.string("mmm"),
                "created_at", DataField.nul()
        ));

        System.out.println(recordJson);
    }

}
