package ru.yandex.chemodan.app.dataapi.utils.serializers.datafieldmap;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.chemodan.app.dataapi.apps.profile.address.AddressTest;
import ru.yandex.chemodan.app.dataapi.apps.profile.events.flights.Flight;
import ru.yandex.chemodan.app.dataapi.apps.profile.events.flights.ProfileFlightsManagerTest;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author tolmalev
 */
public class DataFieldMapSerializerTest {
    @Test
    public void serialize_ClassAggregatesOtherClasses_WritesAllAggregatesAsMap() {
        ListF<ClassB> listOfClasses = Cf.arrayList(new ClassB("classB1"), new ClassB("classB2"));
        DateTime dateTime = new DateTime(2015, 3, 13, 20, 0, DateTimeZone.forOffsetHours(9));
        ClassA aObject = new ClassA(
                "aString",
                new ClassB("bString"),
                dateTime,
                Cf.arrayList("elem1", "elem2"),
                listOfClasses);

        DataFieldMapSerializer<ClassA> serializer = new DataFieldMapSerializer<>(ClassA.class);
        MapF<String, DataField> dataMap = serializer.serialize(aObject);

        DataField aValueDataField = dataMap.getTs("aValue");
        Assert.equals(DataFieldType.STRING, aValueDataField.fieldType);
        Assert.equals("aString", aValueDataField.stringValue());

        DataField dateTimeDataField = dataMap.getTs("dateTime");
        Assert.equals(DataFieldType.DATETIME, dateTimeDataField.fieldType);
        Assert.equals(dateTime, dateTimeDataField.dateTimeValue());

        DataField oneClassB = dataMap.getTs("oneClassB");
        Assert.equals(DataFieldType.MAP, oneClassB.fieldType);
        MapF<String, DataField> classBMap = Cf.hashMap();
        classBMap.put("bValue", DataField.string("bString"));
        Assert.equals(classBMap, oneClassB.mapValue());

        DataField listOfStrings = dataMap.getTs("listOfStrings");
        Assert.equals(DataFieldType.LIST, listOfStrings.fieldType);
        Assert.equals(2, listOfStrings.listValue().length());
        Assert.equals("elem1", listOfStrings.listValue().get(0).stringValue());
        Assert.equals("elem2", listOfStrings.listValue().get(1).stringValue());

        DataField listOfClassesDF = dataMap.getTs("listOfClasses");
        Assert.equals(DataFieldType.LIST, listOfClassesDF.fieldType);
        Assert.equals(2, listOfClassesDF.listValue().length());
        Assert.equals(DataFieldType.MAP, listOfClassesDF.listValue().get(0).fieldType);
        Assert.equals("classB1", listOfClassesDF.listValue().get(0).mapValue().getTs("bValue").stringValue());
        Assert.equals("classB2", listOfClassesDF.listValue().get(1).mapValue().getTs("bValue").stringValue());
    }

    @Test
    public void flight() {
        testParseSerialize(ProfileFlightsManagerTest.sampleFlight());
    }

    @Test
    public void address() {
        testParseSerialize(AddressTest.sampleAddress());
    }

    private void testParseSerialize(Object object) {
        DataFieldMapSerializer serializer = new DataFieldMapSerializer(object.getClass());
        Assert.equals(object, serializer.deserialize(serializer.serialize(object)));
    }

    @Test
    @Ignore
    public void flightSerializeLoadTest() {
        System.out.println("Warming up\n");
        flightSerializeLoadTest(1000);

        System.out.println("Real test\n");
        flightSerializeLoadTest(1000000);
    }

    @Test
    @Ignore
    public void flightDeserializeLoadTest() {
        System.out.println("Warming up\n\n");
        flightDeserializeLoadTest(1000);

        System.out.println("Real test\n");
        flightDeserializeLoadTest(1000000);
    }

    private void flightSerializeLoadTest(int iterations) {
        Flight flight = ProfileFlightsManagerTest.sampleFlight();
        Instant start = Instant.now();
        for (int i = 0; i < iterations; i++) {
            flight.toDataMap();
        }

        System.out.println("Simple: " + TimeUtils.secondsStringToNow(start));

        DataFieldMapSerializer<Flight> serializer = new DataFieldMapSerializer<>(Flight.class);
        start = Instant.now();
        for (int i = 0; i < iterations; i++) {
            serializer.serialize(flight);
        }
        System.out.println("Bender: " + TimeUtils.secondsStringToNow(start));

        System.out.flush();
    }

    private void flightDeserializeLoadTest(int iterations) {
        Flight sampleFlight = ProfileFlightsManagerTest.sampleFlight();
        MapF<String, DataField> manualPlainDataMap = sampleFlight.toDataMap();
        Instant start = Instant.now();
        for (int i = 0; i < iterations; i++) {
            Flight.fromData("rec1", manualPlainDataMap);
        }
        System.out.println("Simple: " + TimeUtils.secondsStringToNow(start));

        DataFieldMapSerializer<Flight> serializer = new DataFieldMapSerializer<>(Flight.class);
        MapF<String, DataField> serializedDataFieldMap = serializer.serialize(sampleFlight);
        start = Instant.now();
        for (int i = 0; i < iterations; i++) {
            serializer.deserialize(serializedDataFieldMap);
        }
        System.out.println("Bender: " + TimeUtils.secondsStringToNow(start));

        System.out.flush();
    }

    @BenderBindAllFields
    private static class ClassA {
        public final String aValue;
        public final ClassB oneClassB;
        public final DateTime dateTime;
        public final ListF<String> listOfStrings;
        public final ListF<ClassB> listOfClasses;

        public ClassA(
                String aValue,
                ClassB oneClassB,
                DateTime dateTime,
                ListF<String> listOfStrings,
                ListF<ClassB> listOfClasses)
        {
            this.aValue = aValue;
            this.oneClassB = oneClassB;
            this.dateTime = dateTime;
            this.listOfStrings = listOfStrings;
            this.listOfClasses = listOfClasses;
        }
    }

    @BenderBindAllFields
    private static class ClassB {
        public final String bValue;

        public ClassB(String bValue) {
            this.bValue = bValue;
        }
    }

}
