package ru.yandex.chemodan.app.dataapi.utils.serializers.datafieldmap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.lang.Check;

/**
 * @author tolmalev
 */
public class DataFieldNode implements BenderJsonNode {

    public final DataField dataField;

    public DataFieldNode(DataField dataField) {
        this.dataField = dataField;
    }

    @Override
    public Option<BenderJsonNode> getField(String fieldName) {
        Check.isTrue(dataField.fieldType == DataFieldType.MAP);
        return dataField.mapValue().getO(fieldName).map(DataFieldNode::new);
    }

    @Override
    public boolean isNull() {
        return dataField.fieldType == DataFieldType.NULL;
    }

    @Override
    public boolean isObject() {
        return dataField.fieldType == DataFieldType.MAP;
    }

    @Override
    public boolean isValueNode() {
        return !isObject() && !isArray();
    }

    @Override
    public boolean isNumber() {
        DataFieldType fieldType = dataField.fieldType;
            return fieldType == DataFieldType.INTEGER
                    || fieldType == DataFieldType.DECIMAL
                    || fieldType == DataFieldType.NAN
                    || fieldType == DataFieldType.INFINITY
                    || fieldType == DataFieldType.NEGATIVE_INFINITY;
    }

    @Override
    public boolean isBoolean() {
        return dataField.fieldType == DataFieldType.BOOLEAN;
    }

    @Override
    public boolean isString() {
        return dataField.fieldType == DataFieldType.STRING;
    }

    @Override
    public boolean isArray() {
        return dataField.fieldType == DataFieldType.LIST;
    }

    @Nonnull
    @Override
    public String getValueAsString() {
        return isNull() ? "null" : dataField.value.toString();
    }

    @Nonnull
    @Override
    public String getString() {
        return dataField.stringValue();
    }

    @Override
    @Nullable
    public Number getNumberValueOrNull() {
        switch (dataField.fieldType) {
            case INTEGER:
                return dataField.integerValue();
            case DECIMAL:
            case NAN:
            case NEGATIVE_INFINITY:
            case INFINITY:
                return dataField.decimalValue();
            default:
                return null;
        }
    }

    @Override
    public boolean getBooleanValueOrFalse() {
        return isBoolean() && dataField.booleanValue();
    }

    @Override
    public ListF<BenderJsonNode> getArrayElements() {
        Check.isTrue(dataField.fieldType == DataFieldType.LIST);
        return dataField.listValue().map(DataFieldNode::new);
    }

    @Override
    public ListF<String> getFieldNames() {
        Check.isTrue(dataField.fieldType == DataFieldType.MAP);
        return dataField.mapValue().keys();
    }

    public Instant getInstant() {
        return dataField.timestampValue();
    }

    public DateTime getDateTime() {
        return dataField.dateTimeValue();
    }

}
