package ru.yandex.chemodan.app.dataapi.utils.serializers.datafieldmap;

import java.math.BigDecimal;
import java.util.Stack;

import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;

/**
 * @author tolmalev
 */
public class DataFieldWriter implements BenderJsonWriter {
    private MapF<String, DataField> finalResult;
    private Stack<DataFieldWriterContext> contextStack = new Stack<>();

    private DataFieldWriterContext getCurrentContext() {
        return contextStack.peek();
    }

    @Override
    public void writeObjectStart() {
        contextStack.push(new DataFieldWriterContext());
    }

    public MapF<String, DataField> getResult() {
        return finalResult;
    }

    @Override
    public void writeObjectEnd() {
        DataFieldWriterContext closedContext = contextStack.pop();
        if(contextStack.size() != 0) {
            addValue(DataField.map(closedContext.dataMap));
        } else {
            finalResult = closedContext.dataMap;
        }
    }

    @Override
    public void writeFieldName(String name) {
        getCurrentContext().fieldName = Option.of(name);
    }

    @Override
    public void writeArrayStart() {
        getCurrentContext().array.push(Cf.arrayList());
    }

    @Override
    public void writeArrayEnd() {
        ListF<DataField> closedArray = getCurrentContext().array.pop();
        addValue(DataField.list(closedArray));
    }

    private void addValue(DataField dataField) {
        DataFieldWriterContext currentContext = getCurrentContext();

        if (!currentContext.array.isEmpty()) {
            currentContext.array.peek().add(dataField);
        } else {
            currentContext.dataMap.put(currentContext.fieldName.get(), dataField);
        }
    }

    @Override
    public void writeString(String value) {
        addValue(DataField.string(value));
    }

    @Override
    public void writeNumber(int value) {
        addValue(DataField.integer(value));
    }

    @Override
    public void writeNumber(long value) {
        addValue(DataField.integer(value));
    }

    public void writeInstant(Instant value) {
        addValue(DataField.timestamp(value));
    }

    public void writeDateTime(DateTime dateTime) {
        addValue(DataField.dateTime(dateTime));
    }

    @Override
    public void writeNumber(double value) {
        if (Double.isInfinite(value)) {
            if (value > 0) {
                addValue(DataField.infinity());
            } else {
                addValue(DataField.negativeInfinity());
            }
        } else if (Double.isNaN(value)) {
            addValue(DataField.nan());
        } else {
            addValue(DataField.decimal(value));
        }
    }

    @Override
    public void writeNumber(float value) {
        if (Float.isInfinite(value)) {
            if (value > 0) {
                addValue(DataField.infinity());
            } else {
                addValue(DataField.negativeInfinity());
            }
        } else if (Float.isNaN(value)) {
            addValue(DataField.nan());
        } else {
            addValue(DataField.decimal(value));
        }
    }

    @Override
    public void writeNumber(BigDecimal value) {
        addValue(DataField.decimal(value.longValue()));
    }

    @Override
    public void writeBoolean(boolean value) {
        addValue(DataField.bool(value));
    }

    @Override
    public void writeNull() {
        addValue(DataField.nul());
    }

    public void write(DataField dataField) {
        addValue(dataField);
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() {
    }
}
