package ru.yandex.chemodan.app.dataapi.web;

import org.joda.time.Instant;
import org.junit.After;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function3V;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.GlobalDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSourceTest;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.TestConstants;
import ru.yandex.chemodan.app.dataapi.web.test.ApiTestBase;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class DatabaseApiTest extends ApiTestBase {

    @Autowired
    private DataApiManager dataApiManager;

    @After
    public void after() {
        timeStart();
    }

    private Database createExternalDatabase() {
        return dataApiManager.createDatabase(UserDatabaseSpec.fromUserAndAlias(uid, TestConstants.EXT_DB_ALIAS_RW));
    }

    @Test
    public void getDatabase_ExternalDatabase_Returns200AndExternalDatabase() {
        Database externalDatabase = createExternalDatabase();

        String url = StringUtils.format("/databases/{}?__uid={}&app={}",
                TestConstants.EXT_DB_ALIAS_RW.aliasId(),
                uid.toString(),
                TestConstants.EXT_DB_ALIAS_RW.clientAppName());

        MockHttpServletResponse resp = sendRequestUsual("GET", url, "");
        assertSuccess(resp, externalDatabase);
    }

    @Test
    public void putDelta_ToExternalDatabase_Returns200AndExternalDatabase() {
        Database externalDatabase = createExternalDatabase();

        String url = StringUtils.format("/databases/{}/diffs?__uid={}&app={}&rev=0",
                TestConstants.EXT_DB_ALIAS_RW.aliasId(),
                uid.toString(),
                TestConstants.EXT_DB_ALIAS_RW.clientAppName());

        String content =
                "{\n"
                + "    \"id\":\"A\",\n"
                + "    \"changes\":[\n"
                + "        {\n"
                + "            \"op\":\"set\",\n"
                + "            \"collectionId\":\"B\",\n"
                + "            \"recordId\":\"C\",\n"
                + "            \"data\":{\n"
                + "                \"D\":\"E\"\n"
                + "            }\n"
                + "        }\n"
                + "    ]\n"
                + "}";
        MockHttpServletResponse resp = sendRequestUsual("PUT", url, content);
        assertSuccess(resp, externalDatabase);
    }

    @Test
    public void putDelta_List_Returns200AndRevisions() {
        Database database = dataApiManager.createDatabase(getTestDatabaseSpec());

        String url = StringUtils.format("/databases/{}/diffs?__uid={}&rev=0", database.aliasId(), uid.toString());

        Function<String, String> mkChangeF = (id) -> ""
                + "{\n"
                + "    \"id\":\"" + id + "\",\n"
                + "    \"changes\":[\n"
                + "        {\n"
                + "            \"op\":\"set\",\n"
                + "            \"collectionId\":\"B\",\n"
                + "            \"recordId\":\"C\",\n"
                + "            \"data\":{\n"
                + "                \"D\":\"E\"\n"
                + "            }\n"
                + "        }\n"
                + "    ]\n"
                + "}\n";

        String content = Cf.list("A", "B").map(mkChangeF).mkString("[", ",", "]");

        MockHttpServletResponse resp = sendRequestUsual("PUT", url, content);

        String expectedPart = ""
                + "\"result\":{\n"
                + "    \"diffs\":[\n"
                + "        {\"id\": \"A\", \"newRev\": 1},\n"
                + "        {\"id\": \"B\", \"newRev\": 2}\n"
                + "    ]\n"
                + "}\n";

        assertResponseContains(resp, expectedPart);
    }

    @Test
    public void putDelta_PerRecordSetForced() {
        Database database = dataApiManager.createDatabase(getTestDatabaseSpec());

        String url = StringUtils.format(
                "/databases/{}/diffs?__uid={}&rev=0&revision_check=per-record", database.aliasId(), uid.toString());

        Function<Boolean, String> mkContentF = forced -> ""
                + "{\n"
                + "    \"id\":\"A\",\n"
                + "    \"changes\":[\n"
                + "        {\n"
                + "            \"op\":\"set\",\n"
                + "            \"forced\":" + forced + ",\n"
                + "            \"collectionId\":\"B\",\n"
                + "            \"recordId\":\"C\",\n"
                + "            \"data\":{\n"
                + "                \"D\":\"A\"\n"
                + "            }\n"
                + "        }\n"
                + "    ]\n"
                + "}\n";

        assertSuccess(sendRequestUsual("PUT", url, mkContentF.apply(false)), database);

        Assert.equals(409, sendRequestUsual("PUT", url, mkContentF.apply(false)).getStatus());
        assertSuccess(sendRequestUsual("PUT", url, mkContentF.apply(true)), database);
    }

    @Test
    public void getSnapshot_ExistedDatabase_ReturnsSnapshot() {
        timeStop(new Instant(1427729131700L));

        Database emptyDb = dataApiManager.createDatabase(getTestDatabaseSpec());

        timeStop(new Instant(1427729132095L));
        dataApiManager.applyDelta(emptyDb, RevisionCheckMode.PER_RECORD, DiskDataSourceTest.makeSimpleDelta(100));

        String url = StringUtils.format("/databases/{}/snapshot?__uid={}&limit=1", emptyDb.aliasId(),
                uid.toString());

        MockHttpServletResponse resp = sendRequestUsual("GET", url, "");
        String expectedPart =
                "\"result\":{\n"
                + "    \"rev\":1,\n"
                + "    \"count\":1,\n"
                + "    \"created\":1427729131700,\n"
                + "    \"modified\":1427729132095,\n"
                + "    \"size\":1173,\n"
                + "    \"objects\":[\n"
                + "        {\n"
                + "            \"collectionId\":\"colId\",\n"
                + "            \"recordId\":\"recId\",\n"
                + "            \"data\":{\n"
                + "                \"testValueName\":{\"type\":\"integer\",\"value\":100}\n"
                + "            }\n"
                + "        }\n"
                + "    ],\"handle\":\"";

        assertResponseContains(resp, expectedPart);
    }

    @Test
    public void getSnapshot_listDeltas_CollectionId() {
        Database db = dataApiManager.createDatabase(getTestDatabaseSpec());

        Cf.list("col1", "col2", "col3").forEach(col -> dataApiManager.applyDelta(
                db, RevisionCheckMode.PER_RECORD, DiskDataSourceTest.makeSimpleDelta(100, col)));

        Function<ListF<String>, String> consSnapshotUrl = colIds -> StringUtils.format(
                "/databases/{}/snapshot?__uid={}&collectionId={}", db.aliasId(), uid + "", colIds.mkString(","));

        Function<ListF<String>, String> consDeltasUrl = collIds -> StringUtils.format(
                "/databases/{}/diffs?__uid={}&rev=0&collectionId={}", db.aliasId(), uid + "", collIds.mkString(","));

        Function3V<MockHttpServletResponse, ListF<String>, ListF<String>> assertContainsAndNot =
                (resp, posIds, negIds) -> posIds.zipWith(s -> true).plus(negIds.zipWith(s -> false)).forEach(
                        (col, pos) -> assertResponseContainsOrNot(resp, "\"collectionId\":\"" + col + "\"", pos));

        MockHttpServletResponse resp = sendRequestUsual("GET", consSnapshotUrl.apply(Cf.list("col1", "col2")), "");
        assertContainsAndNot.apply(resp, Cf.list("col1", "col2"), Cf.list("col3"));

        resp = sendRequestUsual("GET", consSnapshotUrl.apply(Cf.list("col3")), "");
        assertContainsAndNot.apply(resp, Cf.list("col3"), Cf.list("col1", "col2"));

        resp = sendRequestUsual("GET", consDeltasUrl.apply(Cf.list("col1", "col2")), "");
        assertContainsAndNot.apply(resp, Cf.list("col1", "col2"), Cf.list("col3"));

        resp = sendRequestUsual("GET", consDeltasUrl.apply(Cf.list("col3")), "");
        assertContainsAndNot.apply(resp, Cf.list("col3"), Cf.list("col1", "col2"));
    }

    private void assertSuccess(MockHttpServletResponse resp, Database database) {
        Assert.equals(200, resp.getStatus());
        Assert.isTrue(new String(resp.getContentAsByteArray()).contains(database.aliasId()));
    }

    @Override
    protected String getNamespace() {
        return ru.yandex.chemodan.util.web.NS.API;
    }

    private UserDatabaseSpec getTestDatabaseSpec() {
        return new UserDatabaseSpec(uid, new GlobalDatabaseRef("dbId"));
    }
}
