package ru.yandex.chemodan.app.dataapi.web;

import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.web.pojo.ListDeltasResult;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;

/**
 * @author tolmalev
 */
@Action(
        value = @Action.Alias(value = "list-deltas", namespace = ru.yandex.chemodan.util.web.NS.API),
        description = "Список изменений с некоторой ревизии")
@Path(value = "/{context:.*/?}databases/{databaseId}/{tail:(diffs|deltas)}", methods = HttpMethod.GET)
public class ListDeltasAction extends DatabaseAction {

    public static final int MAX_LIMIT = 100;

    @RequestParam
    private long rev;
    @RequestParam(required = false)
    private int limit = 100;

    @RequestListParam(required = false)
    private ListF<String> collectionId;

    // Autowire, because it is only for testing purposes
    @Value("${dataapi.deltas.store.limit}")
    private int maxStoredDeltasCount;

    private final AppSettingsRegistry appSettingsRegistry;

    public ListDeltasAction(DataApiManager dataApiManager, AppSettingsRegistry appSettingsRegistry) {
        super(dataApiManager);
        this.appSettingsRegistry = appSettingsRegistry;
    }

    private MasterSlavePolicy getMasterSlavePolicy() {
        return appSettingsRegistry
                .getSettings(app)
                .getSettingsByDbId(Option.of(databaseId))
                .listDeltasPolicy;
    }

    @Override
    public ListDeltasResult execute() throws Exception {
        return MasterSlaveContextHolder.withPolicy(getMasterSlavePolicy(), () -> {
            limit = Math.min(limit, MAX_LIMIT);

            Database database = dataApiManager.getDatabase(databaseSpec());
            // XXX: hack for testing. Remove when auto-remove of deltas is implemented
            if (database.rev - rev > maxStoredDeltasCount) {
                throw new DeltasGoneException();
            }

            ListF<Delta> deltas;
            if (rev == database.rev) {
                deltas = Cf.list();
            } else {
                deltas = dataApiManager.listDeltas(database.spec(), rev, limit);
            }

            deltas = deltas.filter(d -> d.rev.get() <= database.rev);

            if (deltas.size() < Math.min(database.rev - rev, limit)) {
                throw new DeltasGoneException();
            }

            long requiredRev = rev;
            for (Delta delta : deltas) {
                if (requiredRev != delta.rev.get()) {
                    throw new DeltasGoneException();
                }
                requiredRev++;
            }

            // Hack for old deltas, saved without new revision
            deltas = Delta.addTargetRevIfMissing(deltas);

            if (collectionId.isNotEmpty()) {
                deltas = deltas.map(delta -> delta.withCollectionChangesOnly(collectionId));
            }

            ListDeltasResult result = new ListDeltasResult(database.rev - rev, deltas, database.rev, limit);
            addETagHeaderToResponseIfNeeded(result);
            return result;
        });
    }

}
