package ru.yandex.chemodan.app.dataapi.web;

import javax.annotation.Nonnull;

import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefs;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.web.pojo.TakeoutPojo;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.result.json.JsonWriterActionResult;
import ru.yandex.commune.json.write.BenderJsonWriterWrapper;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;

@Action
@Path(value = "/takeout", methods = HttpMethod.POST)
public class PostTakeoutAction extends DataApiActionBase {

    @SpecialParam
    protected ActionInvocationContext context;

    protected final DataApiManager dataApiManager;
    protected final DiskDataSource diskDataSource;
    protected final Tvm2 tvm2;

    @Value("${takeout.db.list}")
    private  String takeoutDbList;

    @Value("${tvm.takeout.client-id}")
    private int tvmTakeoutClientId;

    protected PostTakeoutAction(DataApiManager dataApiManager, DiskDataSource diskDataSource, Tvm2 tvm2) {
        this.dataApiManager = dataApiManager;
        this.diskDataSource = diskDataSource;
        this.tvm2 = tvm2;
    }

    @Override
    public JsonWriterActionResult execute() throws Exception {
        checkTvm();
        final ListF<DatabaseRefs> dbRefsList = parseDbList().mapEntries(this::getDbRefsByApp);
        final ListF<Database> databases = dbRefsList.flatMap(dbRefs -> diskDataSource.listDatabases(user, dbRefs));
        final ListF<Snapshot> snapshots = dataApiManager.takeout(user, databases);
        final TakeoutPojo takeoutPojo = new TakeoutPojo(snapshots);
        return (jw, attr) -> DataApiBenderUtils.mapper().serializeJson(takeoutPojo, new BenderJsonWriterWrapper(jw));
    }

    private MapF<String, ListF<String>> parseDbList() {
        return StringUtils.toTuple2List(takeoutDbList, ";", ",").groupBy1();
    }

    @Nonnull
    private DatabaseRefs getDbRefsByApp(String app, ListF<String> dbList) {
        return new DatabaseRefs(Option.of(app), dbList);
    }

    private void checkTvm() {
        final Option<String> serviceTicket = context.getRequest().getHeader(TvmHeaders.SERVICE_TICKET);
        final boolean validTvmTicket = serviceTicket.isMatch(ticket -> {
            CheckedServiceTicket parsedServiceTicket = tvm2.checkServiceTicket(ticket);
            return parsedServiceTicket.getSrc() == tvmTakeoutClientId;
        });
        if (!validTvmTicket) {
            throw new UnauthorizedException();
        }
    }
}
