package ru.yandex.chemodan.app.dataapi.web.admin;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppDatabaseSettings;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Denis Bakharev
 */
@ActionContainer
public class AppSettingsAdminPage {

    private final AppSettingsRegistry cacheSettingsRegistry;

    public AppSettingsAdminPage(AppSettingsRegistry appDataCacheSettingsRegistry) {
        this.cacheSettingsRegistry = appDataCacheSettingsRegistry;
    }

    @ZAction(defaultAction = true)
    @Path("/app-settings")
    public AppDataCacheSettingsPojo index() {
        return new AppDataCacheSettingsPojo(cacheSettingsRegistry.getAll()
                .sortedBy(db -> db.databaseId.getOrElse(""))
                .sortedBy(db -> db.appName.getOrElse("")));
    }

    @Path(value = "/app-settings/delete", methods = HttpMethod.POST)
    public void deleteSettings(@RequestParam("appName") String appName, @RequestParam("databaseId") String dbId) {
        cacheSettingsRegistry.deleteTypeSettings(convertAppName(appName), convertDatabaseId(dbId));
    }

    @Path(value = "/app-settings/replace", methods = HttpMethod.POST)
    public void replaceSettings(
            @RequestParam("appName") String appName,
            @RequestParam("databaseId") String dbId,
            @RequestParam("useCache") Option<Boolean> useCache,
            @RequestParam("noKeepDeltas") Option<Boolean> noKeepDeltas,
            @RequestParam("noXiva") Option<Boolean> noXiva,
            @RequestParam("haveDumpInYt") Option<Boolean> haveDumpInYt,
            @RequestParam("getSnapshotPolicy") Option<MasterSlavePolicy> getSnapshotPolicy,
            @RequestParam("listDeltasPolicy") Option<MasterSlavePolicy> listDeltasPolicy,
            @RequestParam("databaseSizeLimit") Option<Long> databaseSizeLimit,
            @RequestParam("databaseRecordsCountLimit") Option<Integer> databaseRecordsCountLimit,
            @RequestParam("lbPushLogTypeName") String lbPushLogTypeName)
    {
        AppDatabaseRef databaseRef = new AppDatabaseRef(appName, dbId);
        AppDatabaseSettings currentSettings = cacheSettingsRegistry.getDatabaseSettings(databaseRef);

        AppDatabaseSettings settings = new AppDatabaseSettings(convertAppName(appName), convertDatabaseId(dbId));

        if(databaseRecordsCountLimit.isSome(0)) {
            databaseRecordsCountLimit = Option.empty();
        }
        if(databaseSizeLimit.isSome(0L)) {
            databaseSizeLimit = Option.empty();
        }

        settings.setUseCache(useCache.getOrElse(false));
        settings.setNoKeepDeltas(noKeepDeltas.getOrElse(false));
        settings.setNoXiva(noXiva.getOrElse(false));
        settings.setHaveDumpInYt(haveDumpInYt.getOrElse(false));
        settings.setGetSnapshotPolicy(getSnapshotPolicy.getOrElse(currentSettings.snapshotPolicy));
        settings.setListDeltasPolicy(listDeltasPolicy.getOrElse(currentSettings.listDeltasPolicy));
        settings.setDatabaseSizeLimit(databaseSizeLimit.map(DataSize::fromBytes));
        settings.setDatabaseRecordsCountLimit(databaseRecordsCountLimit);
        settings.setLbPushLogTypeName(StringUtils.notEmptyO(lbPushLogTypeName));

        cacheSettingsRegistry.setSettings(settings);

        throw new ZRedirectException("/app-settings");
    }

    @XmlRootElement(name = "content")
    @BenderBindAllFields
    private static final class AppDataCacheSettingsPojo {
        @BenderPart(name = "element", wrapperName = "elements")
        public ListF<AppDatabaseSettings> list;

        public AppDataCacheSettingsPojo(ListF<AppDatabaseSettings> list) {
            this.list = list;
        }
    }

    private static Option<String> convertDatabaseId(String dbId) {
        return Option.when(!dbId.isEmpty() && !"*".equals(dbId), dbId);
    }

    private static Option<String> convertAppName(String name) {
        return Option.when(!name.isEmpty(), name);
    }
}
