package ru.yandex.chemodan.app.dataapi.web.admin;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldMarshallerUnmarshaller;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChangeMarshallerUnmarshaller;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeMarshallerUnmarshaller;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.MetaUser;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.result.error.ErrorResult;
import ru.yandex.commune.admin.web.AdminBender;
import ru.yandex.commune.admin.web.bender.DateTimeMarshaller;
import ru.yandex.commune.admin.web.bender.DurationMarshaller;
import ru.yandex.commune.admin.web.bender.InstantMarshaller;
import ru.yandex.commune.admin.z.EmptyContentPojo;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.email.bender.EmailMarshaller;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
@ActionContainer
public class DatabaseInfoAdminPage extends DataApiAdminPageBase {

    private final BenderMapper mapper = new BenderMapper(new BenderConfiguration(
            AdminBender.SETTINGS, CustomMarshallerUnmarshallerFactoryBuilder
            .cons()
            .add(DataField.class,
                    new DataFieldMarshallerUnmarshaller(),
                    new DataFieldMarshallerUnmarshaller())
            .add(RecordChange.class,
                    new RecordChangeMarshallerUnmarshaller(),
                    new RecordChangeMarshallerUnmarshaller())
            .add(FieldChange.class,
                    new FieldChangeMarshallerUnmarshaller(),
                    new FieldChangeMarshallerUnmarshaller())
            .add(DateTime.class, new DateTimeMarshaller())
            .add(Instant.class, new InstantMarshaller())
            .add(Duration.class, new DurationMarshaller())
            .add(Email.class, new EmailMarshaller())
            .build()));

    private final DataApiManager dataApiManager;

    private final Option<AccessRoutines> accessRoutines;

    private DatabaseInfoAdminPage(
            UserMetaManager userMetaManager, DataApiManager dataApiManager, Option<AccessRoutines> accessRoutines)
    {
        super(userMetaManager);
        this.dataApiManager = dataApiManager;
        this.accessRoutines = accessRoutines;
    }

    public DatabaseInfoAdminPage(UserMetaManager userMetaManager, DataApiManager dataApiManager) {
        this(userMetaManager, dataApiManager, Option.empty());
    }

    public DatabaseInfoAdminPage(
            UserMetaManager userMetaManager, DataApiManager dataApiManager, AccessRoutines accessRoutines) {
        this(userMetaManager, dataApiManager, Option.of(accessRoutines));
    }

    private ResolvedAccess checkAccess(
            HttpServletRequestX req, Option<String> app, String databaseId, AccessLevel level)
    {
        if (!accessRoutines.isPresent()) {
            return ResolvedAccess.FULL;
        }

        Option<AccessLevel> accessLevel = accessRoutines.get().getAccessByRequest(req, app, databaseId);

        if (!accessLevel.isPresent() || !accessLevel.get().contains(level)) {
            return ResolvedAccess.DENIED;
        }

        return accessLevel.get().contains(AccessLevel.EDIT) ? ResolvedAccess.FULL : ResolvedAccess.READONLY;
    }

    private ErrorResult getForbiddenResult(AccessLevel level) {
        String forbiddenAction = level == AccessLevel.EDIT ? "edit" : "view";
        return new ErrorResult("Access denied", "You are not allowed to " + forbiddenAction + " this database");
    }

    @ZAction(defaultAction = true, file = "database.xsl")
    @Path("/database-info")
    public Object index(
            @RequestParam(value = "uid", required = false)
            String uid,
            @RequestParam(value = "app", ignoreEmpty = true)
            Option<String> app,
            @RequestParam(value = "databaseId", required = false)
            String databaseId,
            @SpecialParam HttpServletRequestX req)
    {
        uid = StringUtils.defaultIfEmpty(uid, "").trim();
        String trimmedDatabaseId = StringUtils.defaultIfEmpty(databaseId, "").trim();

        MetaUser user = getUser(uid);
        Snapshot snapshot = dataApiManager.getSnapshot(getDatabaseSpec(user, app, trimmedDatabaseId));

        ResolvedAccess access = checkAccess(req, app, trimmedDatabaseId, AccessLevel.VIEW_CONTENT);
        if (access.denied()) {
            return getForbiddenResult(AccessLevel.VIEW_CONTENT);
        }

        return new DatabaseInfoPojo(uid, app, trimmedDatabaseId, user, snapshot, access.readonly());
    }

    @ZAction(file = "delta.xsl")
    @Path("/database-info/delta/{rev}")
    public Object getDelta(
            @RequestParam(value = "uid")
            String uid,
            @RequestParam(value = "app", ignoreEmpty = true)
            Option<String> app,
            @RequestParam(value = "databaseId")
            String databaseId,
            @PathParam("rev")
            long rev,
            @SpecialParam HttpServletRequestX req)
    {
        ResolvedAccess access = checkAccess(req, app, databaseId, AccessLevel.VIEW_CONTENT);
        if (access.denied()) {
            return getForbiddenResult(AccessLevel.VIEW_CONTENT);
        }

        Database database = dataApiManager.getDatabase(getDatabaseSpec(uid, app, databaseId));
        return dataApiManager.getDelta(database.spec(), rev);
    }

    private UserDatabaseSpec getDatabaseSpec(String uid, Option<String> app, String databaseId) {
        return getDatabaseSpec(getUser(uid), app, databaseId);
    }

    private UserDatabaseSpec getDatabaseSpec(MetaUser user, Option<String> app, String databaseId) {
        return new UserDatabaseSpec(user.getUserId(), DatabaseRef.cons(app, databaseId));
    }

    @ZAction(engineId = "json")
    @Path(value = "/database-info/delta", methods = HttpMethod.POST)
    public Object applyDelta(
            @RequestParam(value = "uid")
            String uid,
            @RequestParam(value = "app", ignoreEmpty = true)
            Option<String> app,
            @RequestParam(value = "databaseId")
            String databaseId,
            @RequestParam("delta")
            String deltaStr,
            @SpecialParam HttpServletRequestX req)
    {
        ResolvedAccess access = checkAccess(req, app, databaseId, AccessLevel.VIEW_CONTENT);
        if (access.denied()) {
            return getForbiddenResult(AccessLevel.VIEW_CONTENT);
        }

        Delta delta = mapper.parseJson(Delta.class, deltaStr.getBytes());
        return applyDelta(uid, app, databaseId, delta);
    }

    @ZAction(engineId = "json")
    @Path(value = "/database-info/remove-record", methods = {HttpMethod.GET, HttpMethod.POST})
    public Object removeRecord(
            @RequestParam(value = "uid")
            String uid,
            @RequestParam(value = "app", ignoreEmpty = true)
            Option<String> app,
            @RequestParam(value = "databaseId")
            String databaseId,
            @RequestParam(value = "collectionId")
            String collectionId,
            @RequestParam(value = "recordId")
            String recordId,
            @SpecialParam HttpServletRequestX req)
    {
        ResolvedAccess access = checkAccess(req, app, databaseId, AccessLevel.EDIT);
        if (access.denied()) {
            return getForbiddenResult(AccessLevel.EDIT);
        }

        return applyDelta(uid, app, databaseId,
                new Delta(RecordChange.delete(collectionId, recordId)));
    }

    @ZAction(engineId = "json")
    @Path(value = "/database-info/remove-record-field", methods = {HttpMethod.GET, HttpMethod.POST})
    public Object removeRecordField(
            @RequestParam(value = "uid")
            String uid,
            @RequestParam(value = "app", ignoreEmpty = true)
            Option<String> app,
            @RequestParam(value = "databaseId")
            String databaseId,
            @RequestParam(value = "collectionId")
            String collectionId,
            @RequestParam(value = "recordId")
            String recordId,
            @RequestParam(value = "fieldId")
            String fieldId,
            @SpecialParam HttpServletRequestX req)
    {
        ResolvedAccess access = checkAccess(req, app, databaseId, AccessLevel.EDIT);
        if (access.denied()) {
            return getForbiddenResult(AccessLevel.EDIT);
        }

        return applyDelta(uid, app, databaseId,
                new Delta(RecordChange.update(collectionId, recordId, FieldChange.delete(fieldId))));
    }

    @ZAction(engineId = "json")
    @Path(value = "/database-info/put-record-field", methods = {HttpMethod.GET, HttpMethod.POST})
    public Object putRecordField(
            @RequestParam(value = "uid")
            String uid,
            @RequestParam(value = "app", ignoreEmpty = true)
            Option<String> app,
            @RequestParam(value = "databaseId")
            String databaseId,
            @RequestParam(value = "collectionId")
            String collectionId,
            @RequestParam(value = "recordId")
            String recordId,
            @RequestParam(value = "fieldId")
            String fieldId,
            @RequestParam(value = "type")
            DataFieldType type,
            @RequestParam(value = "value")
            String value,
            @SpecialParam HttpServletRequestX req)
    {
        ResolvedAccess access = checkAccess(req, app, databaseId, AccessLevel.EDIT);
        if (access.denied()) {
            return getForbiddenResult(AccessLevel.EDIT);
        }

        return applyDelta(uid, app, databaseId,
                new Delta(RecordChange.update(collectionId, recordId,
                        FieldChange.put(fieldId, DataField.cons(type, value)))));
    }

    @ZAction(engineId = "json")
    @Path(value = "/database-info/create-record", methods = {HttpMethod.GET, HttpMethod.POST})
    public Object createRecord(
            @RequestParam(value = "uid")
            String uid,
            @RequestParam(value = "app", ignoreEmpty = true)
            Option<String> app,
            @RequestParam(value = "databaseId")
            String databaseId,
            @RequestParam(value = "collectionId")
            String collectionId,
            @RequestParam(value = "recordId")
            String recordId,
            @SpecialParam HttpServletRequestX req)
    {
        ResolvedAccess access = checkAccess(req, app, databaseId, AccessLevel.EDIT);
        if (access.denied()) {
            return getForbiddenResult(AccessLevel.EDIT);
        }

        return applyDelta(uid, app, databaseId,
                new Delta(RecordChange.insertEmpty(collectionId, recordId)));
    }

    private EmptyContentPojo applyDelta(String uid, Option<String> app, String databaseId,
            Delta delta)
    {
        UserDatabaseSpec databaseSpec = getDatabaseSpec(uid, app, databaseId);
        Database database = dataApiManager.getDatabase(databaseSpec);
        dataApiManager.applyDelta(databaseSpec, database.rev, RevisionCheckMode.PER_RECORD, delta);
        return new EmptyContentPojo();
    }

    @Bendable
    @XmlRootElement(name = "content")
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    private static final class DatabaseInfoPojo {
        public final String uid;
        public final Option<String> app;
        public final String databaseId;

        public final MetaUser user;

        public final AdminDatabaseSnapshotPojo snapshot;

        public final boolean readonly;

        private DatabaseInfoPojo(String uid,
                                 Option<String> app,
                                 String databaseId,
                                 MetaUser user,
                                 Snapshot snapshot,
                                 boolean readonly)
        {
            this.uid = uid;
            this.app = app;
            this.databaseId = databaseId;
            this.user = user;
            this.snapshot = new AdminDatabaseSnapshotPojo(
                    snapshot.records()
                            .groupBy(DataRecord::getCollectionId)
                            .mapEntries((colId, records) ->
                                    new AdminCollectionSnapshotPojo(colId, records.map(AdminSnapshotRecordPojo::new))
                            )
            );
            this.readonly = readonly;
        }
    }

    @BenderBindAllFields
    private static final class AdminDatabaseSnapshotPojo {
        @BenderPart(name = "collection", wrapperName = "collections")
        public final ListF<AdminCollectionSnapshotPojo> collections;

        private AdminDatabaseSnapshotPojo(ListF<AdminCollectionSnapshotPojo> collections) {
            this.collections = collections;
        }
    }

    @BenderBindAllFields
    private static final class AdminCollectionSnapshotPojo {
        public final String collectionId;
        @BenderPart(name = "record", wrapperName = "records")
        public final ListF<AdminSnapshotRecordPojo> records;

        private AdminCollectionSnapshotPojo(String collectionId,
                ListF<AdminSnapshotRecordPojo> records)
        {
            this.collectionId = collectionId;
            this.records = records;
        }
    }

    @BenderBindAllFields
    private static final class AdminSnapshotRecordPojo {
        public final String recordId;
        public final DataSize size;
        public final long rev;
        public final MapF<String, AdminSnapshotFieldPojo> data;

        private AdminSnapshotRecordPojo(DataRecord record) {
            recordId = record.getRecordId();
            size = record.getSize();
            rev = record.rev;
            data = record.getData().mapValues(AdminSnapshotFieldPojo::new);
        }
    }

    @BenderBindAllFields
    private static final class AdminSnapshotFieldPojo {

        public final DataFieldType type;
        public final String value;

        public AdminSnapshotFieldPojo(DataField field) {
            this.type = field.fieldType;
            switch (type) {
                case NULL: value = "null"; break;
                case BYTES: value = new String(FastBase64Coder.encode((byte[]) field.value)); break;
                default:
                    value = field.value.toString();
            }
        }
    }

    private enum ResolvedAccess {
        DENIED,
        READONLY,
        FULL;

        public boolean denied() {
            return this == DENIED;
        }

        public boolean readonly() {
            return this == READONLY;
        }
    }
}
