package ru.yandex.chemodan.app.dataapi.web.admin;

import com.github.fge.jsonschema.core.exceptions.ProcessingException;
import com.github.fge.jsonschema.main.JsonSchemaFactory;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.core.generic.DeletionSettings;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeLocation;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.FilterBuildingContext;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.Order;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.ConversionSettingsLoader;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.FormatConverter;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.NodeConversionSettings;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.EmptyContentPojo;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author Denis Bakharev
 */
@ActionContainer
public class TypeSettingsAdminPage {

    private final TypeSettingsRegistry typeSettingsRegistry;

    public TypeSettingsAdminPage(TypeSettingsRegistry typeSettingsRegistry) {
        this.typeSettingsRegistry = typeSettingsRegistry;
    }

    @ZAction(defaultAction = true)
    @Path("/type-settings")
    public TypeSettingsPojo index() {
        return new TypeSettingsPojo(typeSettingsRegistry.getAllTypeSettings().sortedBy(s -> s.typeName));
    }

    @ZAction(file = "editSettings.xsl")
    @Path(value = "/type-settings/edit", methods = HttpMethod.GET)
    public TypeSettings editSettings(
            @RequestParam("typeName") String typeName)
    {
        return typeSettingsRegistry.getTypeSettings(typeName);
    }

    @Path(value = "/type-settings/delete", methods = HttpMethod.POST)
    public void deleteSettings(
            @RequestParam("typeName") String typeName)
    {
        typeSettingsRegistry.deleteTypeSettings(typeName);
    }

    @Path(value = "/type-settings/create", methods = HttpMethod.POST)
    public EmptyContentPojo createNewSettings(
            @RequestParam("typeName") String typeName)
    {
        String defaultSchema =
                "{\n"
                + "    \"type\":\"object\",\n"
                + "    \"properties\":{\n"
                + "    },\n"
                + "    \"required\":[\n"
                + "    ]\n"
                + "}";

        ListF<String> parts = Cf.list(typeName.split("/"))
                .map(String::trim)
                .filter(StringUtils::isNotEmpty);

        Option<String> appName = parts.firstO();
        String databaseId = "common";
        String collectionId = parts.lastO().getOrElse("colId");

        TypeSettings ts = new TypeSettings(
                defaultSchema,
                typeName,
                "id",
                true,
                true,
                Cf.list(),
                new TypeLocation(appName, databaseId, collectionId),
                Option.empty(),
                true);
        typeSettingsRegistry.setTypeSettings(ts);

        throw redirectToEdit(typeName);
    }

    @Path(value = "/type-settings/save", methods = HttpMethod.POST)
    public EmptyContentPojo saveSettings(
            @RequestParam(value = "oldTypeName") String oldTypeName,
            @RequestParam(value = "jsonSchema") String jsonSchema,
            @RequestParam(value = "typeName") String typeName,
            @RequestParam(value = "idPropertyName") String idPropertyName,
            @RequestParam(value = "orderedId", required = false) boolean orderedId,
            @RequestParam(value = "isAllowInsert", required = false) boolean isAllowInsert,
            @RequestParam(value = "isAllowSet", required = false) boolean isAllowSet,
            @RequestParam(value = "isDryRun", required = false) boolean isDryRun,
            @RequestListParam(value = "allowedOrders", splitBy = "\r\n") ListF<String> orders,
            @RequestParam(value = "app") String app,
            @RequestParam(value = "collectionId") String collectionId,
            @RequestParam(value = "databaseId") String databaseId,
            @RequestParam(value = "datePaths") String datePaths,
            @RequestParam(value = "deletionIntervalMs") String deletionIntervalMs)
    {
        Option<String> appO;
        if (StringUtils.isBlank(app)) {
            appO = Option.empty();
        } else {
            appO = Option.of(app);
        }

        Option<DeletionSettings> deletionSettings;
        if (StringUtils.isNotBlank(datePaths) && StringUtils.isNotBlank(deletionIntervalMs)) {
            deletionSettings = Option.of(new DeletionSettings(
                    datePaths, Duration.millis(Long.parseLong(deletionIntervalMs))));
        } else {
            deletionSettings = Option.empty();
        }

        try {
            JsonSchemaFactory.byDefault().getJsonSchema(JsonNodeUtils.getNode(jsonSchema));
            NodeConversionSettings nodeConversionSettings = ConversionSettingsLoader.loadFromJsonSchema(jsonSchema);
            Option<NodeConversionSettings> settingsO = nodeConversionSettings.children.getO(idPropertyName);
            Validate.notEmpty(settingsO, "idProperty was not specified");
            Validate.isTrue(settingsO.get().jsonType.equals("string"), "idProperty should be string");
        } catch (ProcessingException e) {
            throw ExceptionUtils.translate(e);
        }

        TypeSettings ts = new TypeSettings(
                jsonSchema,
                typeName,
                idPropertyName,
                orderedId,
                isAllowInsert,
                isAllowSet,
                orders.map(Order::benderParse),
                new TypeLocation(appO, databaseId, collectionId),
                deletionSettings,
                Option.empty(),
                isDryRun);

        FilterBuildingContext context = new FilterBuildingContext(ts, new FormatConverter(ts.jsonSchema), Option.empty());
        ts.allowedOrders.forEach(o -> o.buildOrder(context));

        typeSettingsRegistry.setTypeSettings(ts);
        throw redirectToEdit(typeName);
    }

    private ZRedirectException redirectToEdit(String typeName) {
        return new ZRedirectException("/type-settings/edit?type-name=" + UrlUtils.urlEncode(typeName));
    }

    @XmlRootElement(name = "content")
    @BenderBindAllFields
    private static final class TypeSettingsPojo {
        @BenderPart(name = "element", wrapperName = "elements")
        public ListF<TypeSettings> list;

        public TypeSettingsPojo(ListF<TypeSettings> list) {
            this.list = list;
        }
    }
}
