package ru.yandex.chemodan.app.dataapi.web.batch;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.JsonNode;
import org.springframework.mock.web.MockHttpServletResponse;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.web.rest.ProxyBatchInterceptor;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.chemodan.util.web.interceptors.WithThreadLocalCache;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.intercept.InterceptWith;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.bind.BoundByBender;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.support.PublicCloneableActionSupport;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.support.tl.ThreadLocalHandle;

/**
 * @author tolmalev
 */
@Action(
        value = @Action.Alias(value = "get-batch", namespace = ru.yandex.chemodan.util.web.NS.API),
        description = "Выполнить batch запрос для получения данных")
@Path(value = "/batch", methods = {HttpMethod.POST})
@WithThreadLocalCache
@InterceptWith({ProxyBatchInterceptor.class})
public class BatchAllAction extends PublicCloneableActionSupport {

    private final Logger logger = LoggerFactory.getLogger(BatchAllAction.class);

    @BoundByBender
    protected BatchRequest batchRequest;

    @SpecialParam
    protected ActionInvocationContext context;

    private final BatchSupport batchSupport;

    public BatchAllAction(BatchSupport batchSupport) {
        this.batchSupport = batchSupport;
    }

    @Override
    public Object execute() throws Exception {
        return execute(batchRequest);
    }

    public BatchResult execute(BatchRequest batchRequest) throws Exception {

        List<OneInvocationJsonResult> invocationResult;
        Stream<BatchRequestItem> stream = batchRequest.items.stream();
        if (batchRequest.items.length() < 2) {
            invocationResult = execSync(stream);
        } else {
            invocationResult = execAsync(stream);
        }
        return new BatchResult(invocationResult);
    }

    private List<OneInvocationJsonResult> execAsync(Stream<BatchRequestItem> stream) {
        Option<String> ycrid = YandexCloudRequestIdHolder.getO();
        return batchSupport.executeStreamAsync(stream.map(item -> () -> sendRequestAndGetResponse(item, ycrid)));
    }

    private List<OneInvocationJsonResult> execSync(Stream<BatchRequestItem> stream) {
        return stream.map(item -> sendRequestAndGetResponse(item, Option.empty())).collect(Collectors.toList());
    }

    OneInvocationJsonResult sendRequestAndGetResponse(BatchRequestItem requestItem, Option<String> ycrid) {

        MockHttpServletResponse response = new MockHttpServletResponse();
        Option<ThreadLocalHandle> tlh = ycrid.map(YandexCloudRequestIdHolder::setAndPushToNdc);
        try {
            batchSupport.service(requestItem, context, response);
            String contentAsString = response.getContentAsString();
            JsonNode responseJsonResult = JsonNodeUtils.getNode(contentAsString);
            JsonNode headers = JsonNodeUtils.getNode(batchSupport.convertToMap(response));
            return new OneInvocationJsonResult(responseJsonResult, headers, response.getStatus());
        } catch (Exception e) {
            logger.info("can't process request: {}", requestItem, e);
            throw ExceptionUtils.translate(e);
        } finally {
            tlh.ifPresent(ThreadLocalHandle::popSafely);
        }
    }

    public int getRequestCount() {
        return Option.ofNullable(batchRequest)
                .map(BatchRequest::getItems)
                .map(List::size)
                .orElse(1);
    }
}
