package ru.yandex.chemodan.app.dataapi.web.batch;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.joda.time.Instant;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.dataapi.apps.profile.address.Address;
import ru.yandex.chemodan.app.dataapi.apps.profile.address.AddressManager;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.generic.GenericObjectManager;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.web.test.ApiTestBase;
import ru.yandex.chemodan.cloud.auth.config.PlatformClient;
import ru.yandex.chemodan.cloud.auth.config.ZkPlatformClientRepository;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class BatchApiTest extends ApiTestBase {

    @Autowired
    private AddressManager addressManager;

    @Autowired
    private GenericObjectManager genericObjectManager;

    @Autowired
    private ZkPlatformClientRepository repository;

    @Autowired
    private TypeSettingsRegistry typeSettingsRegistry;

    private TypeSettings ts;
    private Address address;

    @Before
    public void before() {
        super.before();
        timeStop(new Instant(1433851713265L));

        ts = getTypeSettings();
        typeSettingsRegistry.setTypeSettings(ts);
        address = new Address("addrId");
        PlatformClient platformClient = new PlatformClient();
        platformClient.setEnabled(true);
        platformClient.setOauthScopes(Cf.arrayList("cloud_api.profile:generic.morda.desktopnotifications.read",
                "cloud_api.profile:generic.morda.desktopnotifications.write",
                "cloud_api.profile:generic.extracted_addresses.write",
                "cloud_api.profile:generic.extracted_addresses.read",
                "cloud_api.profile:generic.morda.usersettings.read",
                "cloud_api.profile:generic.morda.usersettings.write",
                "cloud_api.profile:generic.test.read",
                "cloud_api.profile:generic.addresses.write",
                "cloud_api.profile:generic.addresses.read"));
        repository.setClientsByToken(Cf.map("c14e9620915645ba832f3a09264067e2", platformClient));
    }

    @After
    public void after() {
        timeStart();
    }

    @Test
    public void invokeBatchAction_WithExistedObjectsAndReadFromMaster_ReturnsExpectedObjects() {
        addressManager.createAddress(uid, address);
        genericObjectManager.set(uid, "objId", ts.typeName, "{\"requiredProp\":\"value\"}");

        MockHttpServletResponse response = sendRequestUsual(
                "GET", "/batch-new?actions=profile/addresses/addrId,"
                        + "api/generic_data/objId?type_name=my%2Ftype%2FName"
                        + "&read_from_master=true&__uid=" + uid.toString(), "");

        assertContains(response, "invocationInfo");

        assertContains(response,
                "\"result\":["
                        + "{\"code\":200,\"result\":{\"title\":\"no title\",\"latitude\":0.0,\"longitude\":0.0,"
                        + "\"address_id\":\"addrId\",\"created\":1433851713265,\"modified\":1433851713265,"
                        + "\"last_used\":1433851713265,\"tags\":[],\"mined_attributes\":[]}},"

                        + "{\"code\":200,\"result\":{\"requiredProp\":\"value\",\"key\":\"objId\"}}]");
    }

    @Test
    public void invokeBatchAction_WithNotExistedObject_ReturnsError() {
        MockHttpServletResponse response = sendRequestUsual(
                "GET",
                "/batch-new?actions=api/generic_data/unknownId?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), "");

        assertContains(response, "invocationInfo");
        assertContains(response, "\"result\":[{\"code\":404,"
                + "\"error\":{\"name\":\"not-found\",\"message\":\"Record not found: unknownId\",\"stackTrace\":");
    }

    @Test
    public void invokeBatchAllAction() {
        addressManager.createAddress(uid, address);
        genericObjectManager.set(uid, "objId", ts.typeName, "{\"requiredProp\":\"value\"}");

        List<String> requests = IntStream.range(0, 5).mapToObj(i ->
                "{\"relative_url\": \"api/generic_data/objId?type_name=my%2Ftype%2FName"
                        + "&read_from_master=true&__uid=" + uid.toString()
                        + "\", \"headers\":{\"x-inswx\": " + i + "},"
                        + "\"body\":{}, \"method\": \"GET\"}")
                .collect(Collectors.toList());
        MockHttpServletResponse response = sendRequestUsual(
                "POST", "/batch", "{"
                        + "\"items\" : ["
                        + String.join(",", requests)
                        + "]"
                        + "}", Cf.map("hostname", "localhost"));

        assertContains(response, "invocationInfo");

        assertContains(response,
                "\"result\":[{\"code\":200,\"result\":{\"requiredProp\":\"value\",\"key\":\"objId\"}},"
                        + "{\"code\":200,\"result\":{\"requiredProp\":\"value\",\"key\":\"objId\"}},"
                        + "{\"code\":200,\"result\":{\"requiredProp\":\"value\",\"key\":\"objId\"}},"
                        + "{\"code\":200,\"result\":{\"requiredProp\":\"value\",\"key\":\"objId\"}},"
                        + "{\"code\":200,\"result\":{\"requiredProp\":\"value\",\"key\":\"objId\"}}]}");
    }

    /**
     * curl -v -X POST "http://dataapi01f.dst.yandex.net:21859/api/batch"
     * -d '{"items":[{"headers":{"Content-type":"application/json"},
     * "method":"GET","relative_url":"/profile/batch?actions=addresses&__uid=503881259"}]}'
     * @return
     */
    @Test
    public void invokeBatchAllProfile() {
        addressManager.createAddress(uid, address);
        genericObjectManager.set(uid, "objId", ts.typeName, "{\"requiredProp\":\"value\"}");

        String requests = "{\"relative_url\": \"profile/batch?actions=addresses&__uid=" + uid.toString()
                + "\", \"body\":{}, \"method\": \"GET\"}";
        MockHttpServletResponse response = sendRequestUsual(
                "POST", "/batch", "{"
                        + "\"items\" : ["
                        + requests
                        + "]"
                        + "}", Cf.map("hostname", "localhost"));

        assertContains(response, "invocationInfo");

        assertContains(response,
                "\"result\":[{\"code\":200");
    }

    /**
     * /v1/personality/profile/morda/desktopnotifications -> /api/process_generic_data/?http_method=GET&resource_path=morda/desktopnotifications
     * /v1/personality/profile/morda/usersettings         -> /api/process_generic_data/?http_method=GET&resource_path=morda/usersettings
     * /v1/personality/profile/extracted_addresses        -> /api/process_generic_data/?http_method=GET&resource_path=extracted_addresses
     * /v1/personality/profile/addresses/morda            -> /profile/batch?actions=addresses,geopoints&
     */
    @Test
    public void invokeBatchAllProxyTestReal() {

        addressManager.createAddress(uid, address);
        genericObjectManager.set(uid, "objId", ts.typeName, "{\"requiredProp\":\"value\"}");

        List<String> requests = IntStream.range(0, 2).mapToObj(i ->
                "{\"relative_url\": \"/v1/personality/profile/addresses/collectionId?limit=5&offset=0"
                        + "\", \"headers\":{\"x-inswx\": " + i + "},"
                        + "\"body\":{}, \"method\": \"GET\"}")
                .collect(Collectors.toList());
        MockHttpServletResponse response = sendRequest(
                "POST",  "", "/platform/batch", "{"
                        + "\"items\" : ["
                        + String.join(",", requests)
                        + "]"
                        + "}", Cf.map("hostname", "localhost",
                        "Authorization", "ClientToken token=c14e9620915645ba832f3a09264067e2;uid="
                                + uid.toString() + ";"));

        assertContains(response, "total");

        assertContains(response, "\"code\":200");

        assertContains(response, "{\\\"items\\\":[{\\\"data_key\\\":\\\"addrId\\\",\\\"title\\\":\\\"no title\\\",\\\"tags\\\":[],\\\"latitude\\\":0.0,\\\"longitude\\\":0.0}],\\\"limit\\\":5,\\\"total\\\":1,\\\"offset\\\":0}\",\"code\":200},{\"body\":\"{\\\"items\\\":[{\\\"data_key\\\":\\\"addrId\\\",\\\"title\\\":\\\"no title\\\",\\\"tags\\\":[],\\\"latitude\\\":0.0,\\\"longitude\\\":0.0}],\\\"limit\\\":5,\\\"total\\\":1,\\\"offset\\\":0}");
    }

    @Test
    public void invokeBatchAllProxyTestRealError() {

        addressManager.createAddress(uid, address);
        genericObjectManager.set(uid, "objId", ts.typeName, "{\"requiredProp\":\"value\"}");

        List<String> requests = IntStream.range(0, 2).mapToObj(i ->
                "{\"relative_url\": \"/v1/personality/profile/test/collectionId?limit=5&offset=0"
                        + "\", \"headers\":{\"x-inswx\": " + i + "},"
                        + "\"body\":{}, \"method\": \"GET\"}")
                .collect(Collectors.toList());
        MockHttpServletResponse response = sendRequest(
                "POST",  "", "/platform/batch", "{"
                        + "\"items\" : ["
                        + String.join(",", requests)
                        + "]"
                        + "}", Cf.map("hostname", "localhost",
                        "Authorization", "ClientToken token=c14e9620915645ba832f3a09264067e2;uid="
                                + uid.toString() + ";"));

        assertContains(response, "\"code\":400");
        assertContains(response, "unknown-generic-object-type-name");
    }

    @Test
    public void invokeBatchAllProxyTestRealProxySetAddress() {

        addressManager.createAddress(uid, address);

        MockHttpServletResponse response = sendRequest(
                "POST",  "", "/platform/batch", "{"
                        + "\"items\" : ["
                        + "{\"body\":\"{\\\"title\\\":\\\"Дом\\\",\\\"data_key\\\":\\\"home\\\",\\\"latitude\\\":55.7275871665287,\\\"address_line\\\":\\\"Россия, Москва, 4-й Какой-то переулок, 4\\\",\\\"longitude\\\":37.6183377919913,\\\"tags\\\":\\\"_w-_traffic-1-start\\\",\\\"address_line_short\\\":\\\"4-й Какой-то, 4\\\"}\",\"method\":\"PUT\",\"relative_url\":\"/v1/personality/profile/addresses/\"},{\"relative_url\":\"/v1/personality/profile/addresses/\",\"method\":\"PUT\",\"body\":\"{\\\"longitude\\\":37.588144,\\\"tags\\\":\\\"_w-_traffic-1-finish\\\",\\\"address_line_short\\\":\\\"улица Льва Толстого, 16\\\",\\\"title\\\":\\\"Работа\\\",\\\"data_key\\\":\\\"work\\\",\\\"address_line\\\":\\\"Россия, Москва, улица Льва Толстого, 16\\\",\\\"latitude\\\":55.733842}\"}"
                        + "]"
                        + "}", Cf.map("hostname", "localhost",
                        "Authorization", "ClientToken token=c14e9620915645ba832f3a09264067e2;uid="
                                + uid.toString() + ";"));

        assertContains(response, "href");
    }

    @Test
    public void invokeBatchAllProxyTestRealProxySetAddressRawJson() {

        addressManager.createAddress(uid, address);

        MockHttpServletResponse response = sendRequest(
                "POST",  "", "/platform/batch", "{"
                        + "\"items\" : ["
                        + "{\"body\": {\"title\":\"Дом\",\"data_key\":\"home\",\"latitude\":55.7275871665287,\"address_line\":\"Россия, Москва, 4-й Какой-то переулок, 4\",\"longitude\":37.6183377919913,\"tags\":\"_w-_traffic-1-start\",\"address_line_short\":\"4-й Какой-то, 4\"},\"method\":\"PUT\",\"relative_url\":\"http://localhost:8080/v1/personality/profile/addresses/\"},{\"relative_url\":\"http://localhost:8080/v1/personality/profile/addresses/\",\"method\":\"PUT\",\"body\":{\"longitude\":37.588144,\"tags\":\"_w-_traffic-1-finish\",\"address_line_short\":\"улица Льва Толстого, 16\",\"title\":\"Работа\",\"data_key\":\"work\",\"address_line\":\"Россия, Москва, улица Льва Толстого, 16\",\"latitude\":55.733842}}"
                        + "]"
                        + "}", Cf.map("hostname", "localhost",
                        "Authorization", "ClientToken token=c14e9620915645ba832f3a09264067e2;uid="
                                + uid.toString() + ";"));

        assertContains(response, "href");
    }

    @Override
    protected String getNamespace() {
        return ru.yandex.chemodan.util.web.NS.API;
    }
}
