package ru.yandex.chemodan.app.dataapi.web.batch;

import java.util.List;

import com.fasterxml.jackson.databind.JsonNode;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.http.proxy.ProxyManager;
import ru.yandex.chemodan.ratelimiter.yarl.YarlBaseInterceptor;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.chemodan.util.web.interceptors.WithThreadLocalCache;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.support.PublicCloneableActionSupport;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.net.uri.Uri2;

/**
 * @author Denis Bakharev
 */
@Action(
        value = @Action.Alias(value = "get-batch", namespace = ru.yandex.chemodan.util.web.NS.API),
        description = "Выполнить batch запрос для получения данных")
@Path(value = "/batch-new", methods = {HttpMethod.GET})
@WithThreadLocalCache
public class BatchGetAction extends PublicCloneableActionSupport {
    private static final String ACTIONS_PARAM = "actions";
    private static final String ACTIONS_SEPARATOR = ",";

    @SpecialParam
    private ActionInvocationContext context;

    protected final BatchSupport batchSupport;

    public BatchGetAction(BatchSupport batchSupport) {
        this.batchSupport = batchSupport;
    }

    @Override
    public BatchResult execute() throws Exception {
        ListF<String> actions = getActionsFromContext(context).get();

        List<OneInvocationJsonResult> invocationResult = batchSupport.executeStreamAsync(actions
                .stream()
                .map(s -> () -> sendRequestAndGetResponse(s)));

        return new BatchResult(invocationResult);
    }

    private Option<ListF<String>> getActionsFromContext(ActionInvocationContext context) {
        MapF<String, ListF<String>> requestParams = context.getRequest().getParameters();
        return requestParams.getO(ACTIONS_PARAM)
                .map(ListF::first)
                .map(z -> z.split(ACTIONS_SEPARATOR))
                .map(Cf::x)
                .map(paths -> paths
                        .map(path -> StringUtils.removeStart(path, "/"))
                );
    }

    public int getRequestCount(ActionInvocationContext context) {
        return getActionsFromContext(context)
                .map(List::size)
                .orElse(1);
    }

    private OneInvocationJsonResult sendRequestAndGetResponse(String action) {
        String namespace = StringUtils.substringBefore(action, "/");
        String urlWithoutNamespace = StringUtils.substringAfter(action, "/");
        Uri2 uri = Uri2.parse("http://localhost/" + urlWithoutNamespace);
        Tuple2List<String, String> queryArgs = uri.getQueryArgs().plus(getRequestParamsWithoutActions().entries());
        uri = uri.withQueryArgs(queryArgs);
        MockHttpServletRequest request = getRequest(uri);
        MockHttpServletResponse response = new MockHttpServletResponse();

        try {
            batchSupport.service(namespace, request, response, context);
            String contentAsString = response.getContentAsString();
            JsonNode responseJsonResult = JsonNodeUtils.getNode(contentAsString);
            JsonNode headers = JsonNodeUtils.getNode(batchSupport.convertToMap(response));
            return new OneInvocationJsonResult(responseJsonResult, headers, response.getStatus());
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private MockHttpServletRequest getRequest(Uri2 uri) {
        MockHttpServletRequest request = new MockHttpServletRequest("GET", uri.toString());
        request.setAttribute(ProxyManager.SKIP_PROXY, "1");
        request.setAttribute(YarlBaseInterceptor.RATE_LIMITER_IS_CHECKED, "1");
        request.setParameters(uri.getQueryArgs().toMap());
        request.setContentType("application/json");
        request.setPathInfo(uri.getPath());
        request.setServerName(uri.getHost().get());
        return request;
    }

    private MapF<String, String> getRequestParamsWithoutActions() {
        return context.getRequest().getParameters()
                .filterKeys(key -> !key.equals(ACTIONS_PARAM))
                .mapValues(list -> list.firstO().getOrElse(""));
    }
}
