package ru.yandex.chemodan.app.dataapi.web.batch;

import java.io.IOException;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.databind.node.NullNode;
import lombok.RequiredArgsConstructor;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.http.proxy.ProxyManager;
import ru.yandex.chemodan.log.BatchServletRequest;
import ru.yandex.chemodan.log.BatchServletResponse;
import ru.yandex.chemodan.log.DiskLog4jRequestLog;
import ru.yandex.chemodan.ratelimiter.yarl.YarlBaseInterceptor;
import ru.yandex.commune.a3.ActionApp;
import ru.yandex.commune.a3.action.http.ActionInvocationServlet;
import ru.yandex.commune.a3.action.http.HttpStatus;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.net.uri.Uri2;
import ru.yandex.misc.thread.ParallelStreamSupplier;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author vpronto
 */
@RequiredArgsConstructor
public class BatchSupport {

    private final DiskLog4jRequestLog requestLog = new DiskLog4jRequestLog();
    private final ParallelStreamSupplier streamRunner;

    private final ConcurrentHashMap<String, ActionInvocationServlet> servletsMap = new ConcurrentHashMap<>();

    private final Option<OneInvocationJsonResult> defaultValue = getDefaultValue(HttpStatus.INTERNAL_SERVER_ERROR);
    private final Option<OneInvocationJsonResult> timeOutValue = getDefaultValue(HttpStatus.SERVICE_UNAVAILABLE);

    private ActionApp actionApp;

    protected Option<OneInvocationJsonResult> getDefaultValue(HttpStatus status) {
        return Option.of(new OneInvocationJsonResult(NullNode.getInstance(),
                NullNode.getInstance(), status.getStatusCode()));
    }

    List<OneInvocationJsonResult> executeStreamAsync(Stream<Supplier<OneInvocationJsonResult>> stream) {
        return streamRunner.supply(stream, defaultValue, timeOutValue).collect(Collectors.toList());
    }

    Map<?, ?> convertToMap(MockHttpServletResponse response) {
        return response.getHeaderNames().stream().collect(Collectors.toMap(h -> h, response::getHeader));
    }

    public ActionInvocationServlet getServlet(String namespace) {
        return servletsMap.computeIfAbsent(namespace, s -> actionApp.createServlet(namespace));
    }

    void log(MockHttpServletRequest req, ActionInvocationContext context, long timeStamp,
            HttpServletResponse resp, long contentCount)
    {
        requestLog.log(new BatchServletRequest(context.getRequest().getHttpServletRequest(), req, timeStamp),
                new BatchServletResponse(resp, contentCount));
    }

    public void setActionApp(ActionApp actionApp) {
        this.actionApp = actionApp;
    }

    public MockHttpServletRequest createRequest(BatchRequestItem requestItem, HttpServletRequestX httpServletRequest) {

        Uri2 uri = Uri2.parse("http://localhost/" + StringUtils.substringAfter(requestItem.getUrl(), "/"));

        MockHttpServletRequest request = new MockHttpServletRequest(requestItem.method.toUpperCase(), uri.toString());

        Enumeration<String> attributeNames = httpServletRequest.getAttributeNames();
        while (attributeNames.hasMoreElements()) {
            String n = attributeNames.nextElement();
            Object exists = request.getAttribute(n);
            if (exists == null) {
                request.setAttribute(n, httpServletRequest.getAttribute(n));
            }
        }
        request.setAttribute(ProxyManager.SKIP_PROXY, "1");
        request.setAttribute(YarlBaseInterceptor.RATE_LIMITER_IS_CHECKED, "1");

        request.setParameters(uri.getQueryArgs().toMap());

        for (Tuple2<String, String> e : requestItem.headers.entries()) {
            request.addHeader(e._1, e._2);
        }

        request.setPathInfo(uri.getPath());
        request.setServerName(uri.getHost().get());

        requestItem.body.ifPresent(b -> request.setContent(b.getBytes()));
        return request;
    }

    public void service(BatchRequestItem requestItem, ActionInvocationContext context, HttpServletResponse response)
            throws ServletException, IOException
    {
        MockHttpServletRequest request = createRequest(requestItem, context.getRequest().getHttpServletRequest());
        service(requestItem.getNameSpace(), request, response, context);
    }

    public void service(String namespace, MockHttpServletRequest request, HttpServletResponse response,
            ActionInvocationContext context)
            throws ServletException, IOException
    {
        long start = System.currentTimeMillis();
        getServlet(namespace).service(request, response);
        int length = 0;
        if (response instanceof MockHttpServletResponse) {
            length = ((MockHttpServletResponse)response).getContentLength();
        }
        log(request, context, start, response, length);
    }
}
