package ru.yandex.chemodan.app.dataapi.web.direct.a3;

import java.math.BigDecimal;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author tolmalev
 */
public class FieldsFilteredBenderJsonWriter implements BenderJsonWriter {

    private final BenderJsonWriter target;
    private WriterWithParent currentWriter;

    public FieldsFilteredBenderJsonWriter(ListF<String> fields, BenderJsonWriter target) {
        this.target = target;
        currentWriter = new FieldFilteredWriter(fields, null);
    }

    public static FieldsFilteredBenderJsonWriter consFromString(String filterStr, BenderJsonWriter target) {
        return new FieldsFilteredBenderJsonWriter(Cf.x(filterStr.split(","))
                .map(StringUtils::trimToEmpty)
                .filter(StringUtils::isNotEmpty),
                target);
    }

    @Override
    public void writeObjectEnd() {
        currentWriter.writeObjectEnd();
    }

    @Override
    public void writeArrayEnd() {
        currentWriter.writeArrayEnd();
    }

    @Override
    public void writeObjectStart() {
        currentWriter.writeObjectStart();
    }

    @Override
    public void writeFieldName(String name) {
        currentWriter.writeFieldName(name);
    }

    @Override
    public void writeArrayStart() {
        currentWriter.writeArrayStart();
    }

    @Override
    public void writeString(String value) {
        currentWriter.writeString(value);
    }

    @Override
    public void writeNumber(int value) {
        currentWriter.writeNumber(value);
    }

    @Override
    public void writeNumber(long value) {
        currentWriter.writeNumber(value);
    }

    @Override
    public void writeNumber(double value) {
        currentWriter.writeNumber(value);
    }

    @Override
    public void writeNumber(float value) {
        currentWriter.writeNumber(value);
    }

    @Override
    public void writeNumber(BigDecimal value) {
        currentWriter.writeNumber(value);
    }

    @Override
    public void writeBoolean(boolean value) {
        currentWriter.writeBoolean(value);
    }

    @Override
    public void writeNull() {
        currentWriter.writeNull();
    }

    @Override
    public void flush() {
        target.flush();
    }

    @Override
    public void close() {
        target.close();
    }

    abstract class WriterWithParent implements BenderJsonWriter {
        public final WriterWithParent parent;

        protected WriterWithParent(WriterWithParent parent) {
            this.parent = parent;
        }
    }

    class NopWriter extends WriterWithParent {

        protected NopWriter(WriterWithParent parent) {
            super(parent);
        }

        @Override
        public void writeObjectStart() {
            currentWriter = new NopWriter(this);
        }

        @Override
        public void writeObjectEnd() {
            currentWriter = parent;
        }

        @Override
        public void writeFieldName(String name) {

        }

        @Override
        public void writeArrayStart() {
            currentWriter = new NopWriter(this);
        }

        @Override
        public void writeArrayEnd() {
            currentWriter = parent;
        }

        @Override
        public void writeString(String value) {

        }

        @Override
        public void writeNumber(int value) {

        }

        @Override
        public void writeNumber(long value) {

        }

        @Override
        public void writeNumber(double value) {

        }

        @Override
        public void writeNumber(float value) {

        }

        @Override
        public void writeNumber(BigDecimal value) {

        }

        @Override
        public void writeBoolean(boolean value) {

        }

        @Override
        public void writeNull() {

        }

        @Override
        public void flush() {

        }

        @Override
        public void close() {

        }
    }

    class WriteAllWriter extends WriterWithParent {
        protected WriteAllWriter(WriterWithParent parent) {
            super(parent);
        }

        @Override
        public void writeObjectEnd() {
            target.writeObjectEnd();
            currentWriter = parent;
        }

        @Override
        public void writeArrayEnd() {
            target.writeArrayEnd();
            currentWriter = parent;
        }

        @Override
        public void writeObjectStart() {
            target.writeObjectStart();
            currentWriter = new WriteAllWriter(this);
        }

        @Override
        public void writeFieldName(String name) {
            target.writeFieldName(name);
        }

        @Override
        public void writeArrayStart() {
            target.writeArrayStart();
            currentWriter = new WriteAllWriter(this);
        }

        @Override
        public void writeString(String value) {
            target.writeString(value);
        }

        @Override
        public void writeNumber(int value) {
            target.writeNumber(value);
        }

        @Override
        public void writeNumber(long value) {
            target.writeNumber(value);
        }

        @Override
        public void writeNumber(double value) {
            target.writeNumber(value);
        }

        @Override
        public void writeNumber(float value) {
            target.writeNumber(value);
        }

        @Override
        public void writeNumber(BigDecimal value) {
            target.writeNumber(value);
        }

        @Override
        public void writeBoolean(boolean value) {
            target.writeBoolean(value);
        }

        @Override
        public void writeNull() {
            target.writeNull();
        }

        @Override
        public void flush() {
            target.flush();
        }

        @Override
        public void close() {
            target.close();
        }
    }

    class FieldFilteredWriter extends WriterWithParent {
        private MapF<String, WriterWithParent> byFieldWriter;
        private WriterWithParent nextWriter = null;
        private boolean inArray = false;

        public FieldFilteredWriter(ListF<String> fields, WriterWithParent parent) {
            super(parent);

            MapF<String, WriterWithParent> simpleFields = fields
                    .filter(field -> !field.contains("."))
                    .zipWith((f) -> new WriteAllWriter(this))
                    .toMap()
                    .uncheckedCast();

            MapF<String, WriterWithParent> complexFields = fields
                    .filter(field -> field.contains("."))
                    .groupBy(field -> StringUtils.substringBefore(field, "."))
                    .mapValues(flds ->
                            new FieldFilteredWriter(flds.map(s -> StringUtils.substringAfter(s, ".")), this));

            byFieldWriter = simpleFields.plus(complexFields);
        }

        public void writeObjectStart() {
            if (nextWriter != null) {
                if (!(nextWriter instanceof NopWriter)) {
                    target.writeObjectStart();
                }
                currentWriter = nextWriter;
            } else {
                target.writeObjectStart();
            }
        }

        public void writeArrayStart() {
            inArray = true;
            if (nextWriter != null) {
                if (!(nextWriter instanceof NopWriter)) {
                    target.writeArrayStart();
                }
                if (nextWriter instanceof NopWriter || nextWriter instanceof WriteAllWriter) {
                    currentWriter = nextWriter;
                }
            } else {
                target.writeArrayStart();
            }
        }

        public void writeObjectEnd() {
            target.writeObjectEnd();
            currentWriter = parent;
        }

        public void writeFieldName(String name) {
            if (byFieldWriter.containsKeyTs(name)) {
                target.writeFieldName(name);
            }
            nextWriter = byFieldWriter.getO(name).getOrElse(() -> new NopWriter(this));
        }

        public void writeArrayEnd() {
            if (!inArray) {
                currentWriter = parent;
            }
            target.writeArrayEnd();
            inArray = false;
        }

        public void writeString(String value) {
            if (nextWriter != null) {
                nextWriter.writeString(value);
            }
            nextWriter = null;
        }

        public void writeNumber(int value) {
            if (nextWriter != null) {
                nextWriter.writeNumber(value);
            }
            nextWriter = null;
        }

        public void writeNumber(long value) {
            if (nextWriter != null) {
                nextWriter.writeNumber(value);
            }
            nextWriter = null;
        }

        public void writeNumber(double value) {
            if (nextWriter != null) {
                nextWriter.writeNumber(value);
            }
            nextWriter = null;
        }

        public void writeNumber(float value) {
            if (nextWriter != null) {
                nextWriter.writeNumber(value);
            }
            nextWriter = null;
        }

        public void writeNumber(BigDecimal value) {
            if (nextWriter != null) {
                nextWriter.writeNumber(value);
            }
            nextWriter = null;
        }

        public void writeBoolean(boolean value) {
            if (nextWriter != null) {
                nextWriter.writeBoolean(value);
            }
            nextWriter = null;
        }

        public void writeNull() {
            if (nextWriter != null) {
                nextWriter.writeNull();
            }
            nextWriter = null;
        }

        public void flush() {
            target.flush();
        }

        public void close() {
            target.close();
        }
    }
}
