package ru.yandex.chemodan.app.dataapi.web.direct.marshallers;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.field.NamedDataField;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotPojo;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotPojoRow;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseMeta;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.web.direct.a3.DirectDataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.web.pojo.DatabasePojo;
import ru.yandex.chemodan.app.dataapi.web.pojo.ListDatabasesPojo;
import ru.yandex.chemodan.app.dataapi.web.pojo.ListDeltasResult;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.test.Assert;

/**
 * @author metal
 */
public class DirectMarshallerTest {
    private static final BenderMapper mapper = DirectDataApiBenderUtils.mapper();

    @Test
    public void databasePojoSerializationTest() {
        Assert.equals(
                "{" +
                    "\"records_count\":2," +
                    "\"created\":\"2016-03-21T16:58:29.345000+00:00\"," +
                    "\"modified\":\"2016-03-21T16:59:38.789000+00:00\"," +
                    "\"database_id\":\"db_id\"," +
                    "\"handle\":\"db_handle\"," +
                    "\"title\":\"descr\"," +
                    "\"revision\":10," +
                    "\"size\":1048576" +
                "}",
                new String(mapper.serializeJson(generateDatabasePojo("db_id", Option.of("descr"))))
        );

        Assert.equals(
                "{" +
                    "\"records_count\":2," +
                    "\"created\":\"2016-03-21T16:58:29.345000+00:00\"," +
                    "\"modified\":\"2016-03-21T16:59:38.789000+00:00\"," +
                    "\"database_id\":\"db_id\"," +
                    "\"handle\":\"db_handle\"," +
                    "\"revision\":10," +
                    "\"size\":1048576" +
                "}",
                new String(mapper.serializeJson(generateDatabasePojo("db_id", Option.empty())))
        );
    }

    @Test
    public void listDatabasesPojoSerializationTest() {
        ListDatabasesPojo listDatabasesPojo = new ListDatabasesPojo(Cf.list(
                getDatabase("db1", Option.empty()),
                getDatabase("db2", Option.empty())
        ), 2, 3, 1);
        Assert.equals(
                "{" +
                    "\"items\":[" +
                        "{" +
                            "\"records_count\":2," +
                            "\"created\":\"2016-03-21T16:58:29.345000+00:00\"," +
                            "\"modified\":\"2016-03-21T16:59:38.789000+00:00\"," +
                            "\"database_id\":\"db1\"," +
                            "\"handle\":\"db_handle\"," +
                            "\"revision\":10," +
                            "\"size\":1048576" +
                        "}," +
                        "{" +
                            "\"records_count\":2," +
                            "\"created\":\"2016-03-21T16:58:29.345000+00:00\"," +
                            "\"modified\":\"2016-03-21T16:59:38.789000+00:00\"," +
                            "\"database_id\":\"db2\"," +
                            "\"handle\":\"db_handle\"," +
                            "\"revision\":10," +
                            "\"size\":1048576" +
                        "}" +
                    "]," +
                    "\"total\":2," +
                    "\"limit\":3," +
                    "\"offset\":1" +
                "}",
                new String(mapper.serializeJson(listDatabasesPojo))
        );
    }

    @Test
    public void snapshotPojoSerializationTest() {
        SnapshotPojo snapshotPojo = getSnapshotPojo(Cf.list("testkey", "keytest"), Option.empty());
        Assert.equals(
                "{" +
                    "\"records_count\":2," +
                    "\"created\":\"2016-03-21T16:58:29.345000+00:00\"," +
                    "\"modified\":\"2016-03-21T16:59:38.789000+00:00\"," +
                    "\"records\":{" +
                        "\"items\":[" +
                            "{" +
                                "\"record_id\":\"recid\"," +
                                "\"collection_id\":\"collid\"," +
                                "\"fields\":[" +
                                    "{" +
                                        "\"field_id\":\"testkey2\"," +
                                        "\"value\":{" +
                                            "\"type\":\"string\"," +
                                            "\"string\":\"test\"" +
                                        "}" +
                                    "}," +
                                    "{" +
                                        "\"field_id\":\"testkey1\"," +
                                        "\"value\":{" +
                                            "\"type\":\"boolean\"," +
                                            "\"boolean\":true" +
                                        "}" +
                                    "}" +
                                "]," +
                                "\"revision\":27" +
                            "}," +
                            "{" +
                                "\"record_id\":\"recid\"," +
                                "\"collection_id\":\"collid\"," +
                                "\"fields\":[" +
                                    "{" +
                                        "\"field_id\":\"keytest2\"," +
                                        "\"value\":{" +
                                            "\"type\":\"string\"," +
                                            "\"string\":\"test\"" +
                                        "}" +
                                    "}," +
                                    "{" +
                                        "\"field_id\":\"keytest1\"," +
                                        "\"value\":{" +
                                            "\"type\":\"boolean\"," +
                                            "\"boolean\":true" +
                                        "}" +
                                    "}" +
                                "]," +
                                "\"revision\":27" +
                            "}" +
                        "]" +
                    "}," +
                    "\"database_id\":\"db_id\"," +
                    "\"handle\":\"snapshot_db_handle\"," +
                    "\"revision\":7," +
                    "\"size\":1048576" +
                "}",
                new String(mapper.serializeJson(snapshotPojo))
        );
    }

    @Test
    public void listDeltasResultSerializationTest() {
        ListDeltasResult listDeltasResult = new ListDeltasResult(2, Cf.list(getDelta()), 3, 4);

        Assert.equals(
                "{" +
                    "\"base_revision\":1," +
                    "\"items\":[" +
                        "{" +
                            "\"base_revision\":12," +
                            "\"delta_id\":\"delta_id\"," +
                            "\"changes\":[" +
                                "{" +
                                    "\"change_type\":\"insert\"," +
                                    "\"collection_id\":\"insert_colid\"," +
                                    "\"record_id\":\"insert_recid\"," +
                                    "\"changes\":[]" +
                                "}," +
                                "{" +
                                    "\"change_type\":\"delete\"," +
                                    "\"collection_id\":\"delete_colid\"," +
                                    "\"record_id\":\"delete_recid\"," +
                                    "\"changes\":[]" +
                                "}," +
                                "{" +
                                    "\"change_type\":\"set\"," +
                                    "\"collection_id\":\"set_colid\"," +
                                    "\"record_id\":\"set_recid\"," +
                                    "\"changes\":[]" +
                                "}," +
                                "{" +
                                    "\"change_type\":\"update\"," +
                                    "\"collection_id\":\"update_colid\"," +
                                    "\"record_id\":\"update_recid\"," +
                                    "\"changes\":[" +
                                        "{" +
                                            "\"field_id\":\"delete_key\"," +
                                            "\"change_type\":\"delete\"" +
                                        "}," +
                                        "{" +
                                            "\"field_id\":\"put_key\"," +
                                            "\"change_type\":\"set\"," +
                                            "\"value\":{" +
                                                "\"type\":\"boolean\"," +
                                                "\"boolean\":true" +
                                            "}" +
                                        "}," +
                                        "{" +
                                            "\"field_id\":\"put_list_item_key\"," +
                                            "\"change_type\":\"list_item_set\"," +
                                            "\"list_item\":65," +
                                            "\"value\":{" +
                                                "\"type\":\"null\"," +
                                                "\"null\":true" +
                                            "}" +
                                        "}," +
                                        "{" +
                                            "\"field_id\":\"delete_list_item_key\"," +
                                            "\"change_type\":\"list_item_delete\"," +
                                            "\"list_item\":78" +
                                        "}," +
                                        "{" +
                                            "\"field_id\":\"insert_list_item_key\"," +
                                            "\"change_type\":\"list_item_insert\"," +
                                            "\"list_item\":3," +
                                            "\"value\":{" +
                                                "\"type\":\"integer\"," +
                                                "\"integer\":57" +
                                            "}" +
                                        "}," +
                                        "{" +
                                            "\"field_id\":\"move_list_item_key\"," +
                                            "\"change_type\":\"list_item_move\"," +
                                            "\"list_item\":98," +
                                            "\"list_item_dest\":56" +
                                        "}" +
                                    "]" +
                                "}" +
                            "]," +
                            "\"revision\":23" +
                        "}" +
                    "]," +
                    "\"total\":2," +
                    "\"limit\":4," +
                    "\"revision\":3" +
                "}",
                new String(mapper.serializeJson(listDeltasResult))
        );
    }

    @Test
    public void dataFieldsSerializationTest() {
        ListF<DataField> dataFields = Cf.arrayList();
        dataFields.add(DataField.bytes("bytesstring".getBytes()));
        dataFields.add(DataField.string("valuestring"));
        dataFields.add(DataField.decimal(123.456));
        dataFields.add(DataField.list(DataField.bool(false), DataField.integer(963)));
        dataFields.add(DataField.timestamp(Instant.parse("2016-03-29T12:47:34.642Z")));
        dataFields.add(DataField.dateTime(new DateTime(Instant.parse("2016-02-15T21:45:32Z"))));
        dataFields.add(DataField.integer(789));
        dataFields.add(DataField.bool(true));
        dataFields.add(DataField.nan());
        dataFields.add(DataField.negativeInfinity());
        dataFields.add(DataField.infinity());
        dataFields.add(DataField.nul());
        TestClassWithDataFields obj = new TestClassWithDataFields(dataFields);

        Assert.equals(
                "{" +
                    "\"fields\":[" +
                        "{" +
                            "\"type\":\"binary\"," +
                            "\"binary\":\"Ynl0ZXNzdHJpbmc=\"" +
                        "}," +
                        "{" +
                            "\"type\":\"string\"," +
                            "\"string\":\"valuestring\"" +
                        "}," +
                        "{" +
                            "\"type\":\"double\"," +
                            "\"double\":123.456" +
                        "}," +
                        "{" +
                            "\"type\":\"list\"," +
                            "\"list\":[" +
                                "{" +
                                    "\"type\":\"boolean\"," +
                                    "\"boolean\":false" +
                                "}," +
                                "{" +
                                    "\"type\":\"integer\"," +
                                    "\"integer\":963" +
                                "}" +
                            "]" +
                        "}," +
                        "{" +
                            "\"type\":\"datetime\"," +
                            "\"datetime\":\"2016-03-29T12:47:34.642000+00:00\"" +
                        "}," +
                        "{" +
                            "\"type\":\"datetime\"," +
                            "\"datetime\":\"2016-02-15T21:45:32.000000+00:00\"" +
                        "}," +
                        "{" +
                            "\"type\":\"integer\"," +
                            "\"integer\":789" +
                        "}," +
                        "{" +
                            "\"type\":\"boolean\"," +
                            "\"boolean\":true" +
                        "}," +
                        "{" +
                            "\"type\":\"nan\"," +
                            "\"nan\":true" +
                        "}," +
                        "{" +
                            "\"type\":\"ninf\"," +
                            "\"ninf\":true" +
                        "}," +
                        "{" +
                            "\"type\":\"inf\"," +
                            "\"inf\":true" +
                        "}," +
                        "{" +
                            "\"type\":\"null\"," +
                            "\"null\":true" +
                        "}" +
                    "]" +
                "}",
                new String(mapper.serializeJson(obj))
        );
    }

    private Delta getDelta() {
        return new Delta(getRecordChanges(getFieldChanges()), Option.of("delta_id"), Option.of(12L), Option.of(23L));
    }

    private ListF<FieldChange> getFieldChanges() {
        FieldChange delete = FieldChange.delete("delete_key");
        FieldChange put = FieldChange.put("put_key", DataField.bool(true));
        FieldChange putListItem = FieldChange.putListItem("put_list_item_key", 65, DataField.nul());
        FieldChange deleteListItem = FieldChange.deleteListItem("delete_list_item_key", 78);
        FieldChange insertListItem = FieldChange.insertListItem("insert_list_item_key", 3, DataField.integer(57));
        FieldChange moveListItem = FieldChange.moveListItem("move_list_item_key", 98, 56);

        return Cf.list(delete, put, putListItem, deleteListItem, insertListItem, moveListItem);
    }

    private ListF<RecordChange> getRecordChanges(ListF<FieldChange> fieldChanges) {
        RecordChange insert = RecordChange.insertEmpty("insert_colid", "insert_recid");
        RecordChange delete = RecordChange.delete("delete_colid", "delete_recid");
        RecordChange set = RecordChange.set("set_colid", "set_recid", Cf.map());
        RecordChange update = RecordChange.update("update_colid", "update_recid", fieldChanges);

        return Cf.list(insert, delete, set, update);
    }

    private SnapshotPojo getSnapshotPojo(ListF<String> dataFieldKeys, Option<String> description) {
        return new SnapshotPojo(7, "db_id", "snapshot_db_handle",
                getDatabaseMeta(description), dataFieldKeys.map(this::getSnapshotPojoRow));
    }

    private SnapshotPojoRow getSnapshotPojoRow(String key) {
        return new SnapshotPojoRow("collid", "recid", getDataFields(key), 27);
    }

    private DataFields getDataFields(String key) {
        return new DataFields(
                NamedDataField.bool(key + "1", true),
                NamedDataField.string(key + "2", "test")
        );
    }

    private DatabasePojo generateDatabasePojo(String databaseId, Option<String> description) {
        return new DatabasePojo(getDatabase(databaseId, description));
    }

    private Database getDatabase(String databaseId, Option<String> description) {
        DatabaseHandle dbHandle = new DatabaseHandle("app", databaseId, "db_handle");
        return new Database(DataApiUserId.parse("123"), dbHandle, 10, getDatabaseMeta(description));
    }

    private DatabaseMeta getDatabaseMeta(Option<String> description) {
        return new DatabaseMeta(
                    Instant.parse("2016-03-21T16:58:29.345Z"),
                    Instant.parse("2016-03-21T16:59:38.789Z"),
                    DataSize.MEGABYTE, 2, description);
    }

    @BenderBindAllFields
    private static class TestClassWithDataFields {
        @SuppressWarnings("unused")
        private ListF<DataField> fields;

        public TestClassWithDataFields(ListF<DataField> fields) {
            this.fields = fields;
        }
    }
}
