package ru.yandex.chemodan.app.dataapi.web.direct.unmarshallers;

import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.chemodan.app.dataapi.web.direct.DirectDataFieldMapper;
import ru.yandex.chemodan.app.dataapi.web.direct.marshallers.DataFieldJsonSerializers;
import ru.yandex.chemodan.util.bender.ISODateTimeUnmarshaller;
import ru.yandex.chemodan.util.bender.ISOInstantUnmarshaller;
import ru.yandex.chemodan.util.bender.JsonFieldLevelUnmarshaller;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.ParseResult;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.lang.Check;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DataFieldJsonParsers {
    private static final ISOInstantUnmarshaller instantUnmarshaller = new ISOInstantUnmarshaller();

    private static final ISODateTimeUnmarshaller dataTimeUnmarshaller = new ISODateTimeUnmarshaller();

    public static JsonFieldLevelUnmarshaller consUnmarshaller() {
        return (node, context) -> ParseResult.result(parse(node));
    }

    public static DataField parse(BenderJsonNode node) {
        DataFieldType fieldType = parseFieldType(node.getField(DataFieldJsonSerializers.TYPE).get());
        DirectDataFieldMapper mapper = fieldType.directMapper;
        return mapper.parse(
                node.getField(mapper.typeName)
                        .getOrThrow(() -> consFieldMissingException(mapper.typeName))
        );
    }

    public static DataFieldType parseFieldType(BenderJsonNode node) {
        return DataFieldType.byDirectName(node.getValueAsString());
    }

    public static DataField parseBoolean(BenderJsonNode node) {
        checkIsBoolean(node);
        return DataField.bool(node.getBooleanValueOrFalse());
    }

    public static DataField parseInteger(BenderJsonNode node) {
        checkIsNumber(node);
        return DataField.integer(
                getNumber(node)
                        .longValue()
        );
    }

    public static DataField parseDecimal(BenderJsonNode node) {
        checkIsNumber(node);
        return DataField.decimal(
                getNumber(node)
                        .doubleValue()
        );
    }

    private static Number getNumber(BenderJsonNode node) {
        return node.getNumberValueOrNull();
    }

    public static DataField parseString(BenderJsonNode node) {
        chekIsStringNode(node);
        return DataField.string(node.getValueAsString());
    }

    public static DataField parseBytes(BenderJsonNode node) {
        chekIsStringNode(node);
        byte[] bytes = FastBase64Coder.decode(node.getValueAsString());
        return DataField.bytes(bytes);
    }

    public static DataField parseTimestamp(BenderJsonNode node) {
        chekIsStringNode(node);
        Instant value = instantUnmarshaller.convert(node.getValueAsString());
        return DataField.timestamp(value);
    }

    public static DataField parseDateTime(BenderJsonNode node) {
        chekIsStringNode(node);
        DateTime value = dataTimeUnmarshaller.convert(node.getValueAsString());
        return DataField.dateTime(value);
    }

    public static DataField parseList(BenderJsonNode node) {
        checkIsArray(node);
        return DataField.list(
                node.getArrayElements()
                        .map(DataFieldJsonParsers::parse));
    }

    public static DataField parseMap(BenderJsonNode node) {
        checkIsObject(node);
        return DataField.map(
                node.getFieldNames()
                        .toMapMappingToValue(
                                fieldId -> parse(
                                        node.getField(fieldId)
                                                .getOrThrow(() -> consFieldMissingException(fieldId))
                                )
                        ));
    }

    private static IllegalStateException consFieldMissingException(String fieldName) {
        return new IllegalStateException("Error while parsing DataField: mandatory field is missing - " + fieldName);
    }

    private static void checkIsBoolean(BenderJsonNode node) {
        checkIsType(node, BenderJsonNode::isBoolean, "boolean");
    }

    private static void checkIsNumber(BenderJsonNode node) {
        checkIsType(node, BenderJsonNode::isNumber, "number");
    }

    private static void chekIsStringNode(BenderJsonNode node) {
        checkIsType(node, BenderJsonNode::isString, "string");
    }

    private static void checkIsArray(BenderJsonNode node) {
        checkIsType(node, BenderJsonNode::isArray, "array");
    }

    private static void checkIsObject(BenderJsonNode node) {
        checkIsType(node, BenderJsonNode::isObject, "object");
    }

    private static void checkIsType(BenderJsonNode node, Function1B<BenderJsonNode> isTypeF, String typeName) {
        Check.isTrue(isTypeF.apply(node), "Expecting " + typeName + " node");
    }

    private DataFieldJsonParsers() {}
}
