package ru.yandex.chemodan.app.dataapi.web.generic;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.GenericObjectManager;
import ru.yandex.chemodan.app.dataapi.core.generic.LimitedResult;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.Condition;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.ConditionParser;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.ObjectsFilter;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.Order;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.OrderParser;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.OrderedUuidField;
import ru.yandex.chemodan.app.dataapi.web.ActionUtils;
import ru.yandex.chemodan.util.web.OkPojo;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.RoundRobinCounterMap;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author Denis Bakharev
 */
public class GenericObjectsApiHelper implements MonicaContainer {

    public static final String READ_FROM_MASTER = "readFromMaster";

    @MonicaMetric(description = "Мониторинг запросов")
    @GroupByDefault
    private final InstrumentMap requests = new InstrumentMap();

    @MonicaMetric(description = "Счетчик ошибок в запросах")
    @GroupByDefault
    private final RoundRobinCounterMap errors = new RoundRobinCounterMap(Duration.standardMinutes(5));

    private final GenericObjectManager genericObjectManager;

    public GenericObjectsApiHelper(GenericObjectManager genericObjectManager) {
        this.genericObjectManager = genericObjectManager;
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "dataapi",
                new MetricName("dataapi", "GenericObjects"),
                "Мониторинг работы GenericObject ручек");
    }

    private <TResult> TResult executeWithMeasureAndPolicy(
            MetricName metricName,
            MasterSlavePolicy policy,
            Function0<TResult> function)
    {
        try {
            return MasterSlaveContextHolder.withPolicy(policy, () -> requests.measure(function, Cf.list(metricName)));
        } catch (Throwable ex) {
            ExceptionUtils.throwIfUnrecoverable(ex);
            errors.inc(1, metricName);
            throw ex;
        }
    }

    public Object handleDelete(DataApiUserId user, String objectId, String typeName) {
        return executeWithMeasureAndPolicy(
                new MetricName(typeName, "delete"),
                MasterSlavePolicy.RW_M,
                () -> {
                    genericObjectManager.delete(user, objectId, typeName);
                    return new OkPojo();
                });
    }

    public Object handleGet(DataApiUserId user, String objectId, String typeName, ActionInvocationContext context) {
        return executeWithMeasureAndPolicy(
                new MetricName(typeName, "get"),
                getReadPolicyFromContext(context),
                () -> {
                    String json = genericObjectManager.get(user, objectId, typeName);
                    return new RawResult(json);
                });
    }

    public Object handlePut(DataApiUserId user, String objectId, String typeName, ActionInvocationContext context) {
        return executeWithMeasureAndPolicy(
                new MetricName(typeName, "put"),
                MasterSlavePolicy.RW_M,
                () -> {
                    Condition ifMatch = getStringParam("rewriteIf", context)
                            .map(ConditionParser::parse).getOrElse(Condition::trueCondition);

                    String json = context.getRequest().getHttpServletRequest().getInputStreamX().readString();
                    String result = genericObjectManager.set(user, objectId, typeName, json, ifMatch);
                    context.getHttpContext().setStatusCode(HttpStatus.SC_200_OK);

                    return new RawResult(result);
                });
    }

    public Object handlePost(DataApiUserId user, String typeName, ActionInvocationContext context) {
        return executeWithMeasureAndPolicy(
                new MetricName(typeName, "post"),
                MasterSlavePolicy.RW_M,
                () -> {
                    String json = context.getRequest().getHttpServletRequest().getInputStreamX().readString();

                    String result = genericObjectManager.insert(user, typeName, json);
                    context.getHttpContext().setStatusCode(HttpStatus.SC_201_CREATED);

                    return new RawResult(result);
                });
    }

    public Object handleGetList(DataApiUserId user, String typeName, ActionInvocationContext context) {
        return executeWithMeasureAndPolicy(
                new MetricName(typeName, "get-list"),
                getReadPolicyFromContext(context),
                () -> {
                    ListF<Option<String>> range = Cf.list("query.range.from", "query.range.to")
                            .map(p -> getStringParam(p, context));

                    Condition condition = range.exists(Option::isPresent)
                            ? OrderedUuidField.F.between(range.first(), range.last())
                            : getStringParam("where", context).map(ConditionParser::parse).getOrElse(Condition::trueCondition);

                    Order order = getBoolParam("query.order.asc", context).map(OrderedUuidField.F::orderBy)
                            .orElse(() -> getStringParam("orderBy", context).map(OrderParser::parse))
                            .getOrElse(Order::empty);

                    Option<Integer> limit = getIntParam("limit", context);
                    Option<Integer> offset = getIntParam("offset", context);

                    boolean requestTotalCount = getBoolParam("requestTotalCount", context).getOrElse(true);

                    SqlLimits limits = ActionUtils.getLimits(limit, offset);
                    LimitedResult<String> limitedResult = genericObjectManager.getList(
                            user, typeName, new ObjectsFilter(condition, order, limits), requestTotalCount);

                    String itemsString = "\"items\":[" + String.join(",", limitedResult.result) + "]";
                    String selectionStats = "";

                    if (limitedResult.totalCount.isPresent() && limits != SqlLimits.all()) {
                        int total = limitedResult.totalCount.get();

                        selectionStats = StringUtils.format(
                                ",\"total\":{},\"limit\":{},\"offset\":{}",
                                total,
                                limits.getCount(),
                                limits.getFirst());
                    }
                    return new RawResult("{" + itemsString + selectionStats + "}");
                });
    }

    private MasterSlavePolicy getReadPolicyFromContext(ActionInvocationContext context) {
        Option<Boolean> isReadFromMaster =
                context.getRequest().getParameter(READ_FROM_MASTER).firstO().map(Boolean::valueOf);
        if(isReadFromMaster.isPresent() && isReadFromMaster.get()) {
            return MasterSlavePolicy.R_M;
        }

        return MasterSlavePolicy.R_ANY;
    }

    private Option<Integer> getIntParam(String paramName, ActionInvocationContext context) {
        return context.getRequest().getParameter(paramName).firstO().map(Integer::parseInt);
    }

    private Option<Boolean> getBoolParam(String paramName, ActionInvocationContext context) {
        return context.getRequest().getParameter(paramName).firstO().map(Boolean::valueOf);
    }

    private Option<String> getStringParam(String paramName, ActionInvocationContext context) {
        return context.getRequest().getParameter(paramName).firstO();
    }
}
