package ru.yandex.chemodan.app.dataapi.web.generic;

import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.Function3;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.OrderedUUID;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.stubs.UserMetaManagerStub;
import ru.yandex.chemodan.app.dataapi.web.test.ApiTestBase;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class GenericObjectsApiTest extends ApiTestBase {

    @Autowired
    private TypeSettingsRegistry typeSettingsRegistry;

    @Autowired
    private DataApiManager dataApiManager;

    private boolean isBeforeInvoked;
    private TypeSettings typeSettings;

    @Before
    public void before() {
        super.before();

        if (isBeforeInvoked) {
            return;
        }

        typeSettings = getTypeSettings();

        typeSettingsRegistry.setTypeSettings(typeSettings);
        isBeforeInvoked = true;
    }

    @Test
    public void processGenericObject_GetNotExistedObject_Returns404AndRecordNotFound() {
        String json = "{\"key\":\"objId\", \"requiredProp\":\"value\"}";
        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?http_method=GET&resource_path=my%2Ftype%2FName%2FObjId&__uid=" + uid.toString(),
                json);

        Assert.equals(404, response.getStatus());
        assertContains(response, "\"error\":{\"name\":\"not-found\",\"message\":\"Record not found: ObjId\"");
    }

    @Test
    public void processGenericObject_PutMethod_SetObject() {
        String json = "{\"requiredProp\":\"value\"}";
        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?http_method=PUT&resource_path=my%2Ftype%2FName%2FobjId&__uid=" + uid.toString(),
                json);

        Assert.equals(200, response.getStatus());
        assertContains(response, "{\"requiredProp\":\"value\",\"key\":\"objId\"}");
    }

    @Test
    public void processGenericObject_InvalidJsonData_ReturnsBadRequest() {
        String json = "{\"key\":\"objId\", \"require";
        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?http_method=PUT&resource_path=my%2Ftype%2FName%2FobjId&__uid=" + uid.toString(),
                json);

        Assert.equals(400, response.getStatus());
    }

    @Test
    public void processGenericObject_PostMethod_InsertObject() {
        String json = "{\"requiredProp\":\"value\"}";
        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?http_method=POST&resource_path=my%2Ftype%2FName&__uid=" + uid.toString(),
                json);

        Assert.equals(201, response.getStatus());
        assertContains(response, "\"requiredProp\":\"value\"");
    }

    @Test
    public void processGenericObject_TypeNameDoesNotExist_Returns400() {
        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?http_method=GET&resource_path=UnknownTypeName%2FobjId1&__uid=" + uid.toString(),
                "");

        Assert.equals(400, response.getStatus());
    }

    @Test
    public void processGenericObject_GetMethodWithId_ReturnsOneObject() {
        createObject("objId1", "propValue1");

        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?http_method=GET&resource_path=my%2Ftype%2FName%2FobjId1&__uid=" + uid.toString(),
                "");

        Assert.equals(200, response.getStatus());
        assertContains(response, "{\"requiredProp\":\"propValue1\",\"key\":\"objId1\"}");
    }

    @Test
    public void processGenericObject_DeleteMethod_DeleteObject() {
        createObject("objId1", "propValue1");

        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?http_method=DELETE&resource_path=my%2Ftype%2FName%2FobjId1&__uid=" + uid.toString(),
                "");

        Assert.equals(200, response.getStatus());

        Assert.isEmpty(getDataRecord("objId1"));
    }

    @Test
    public void processGenericObject_GetMethodWithoutIdAndSetOffsetAndLimit_ReturnsMultipleObjectAndOffsetLimitTotalValue() {
        createObject("objId1", "propValue1");
        createObject("objId2", "propValue2");

        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?offset=0&limit=10&http_method=GET&resource_path=my%2Ftype%2FName&__uid=" + uid.toString(),
                "");

        Assert.equals(200, response.getStatus());

        String expectedString =
                "\"result\":{\"items\":["
                + "{\"requiredProp\":\"propValue1\",\"key\":\"objId1\"},"
                + "{\"requiredProp\":\"propValue2\",\"key\":\"objId2\"}"
                + "],\"total\":2,\"limit\":10,\"offset\":0}";
        assertContains(response, expectedString);
    }

    @Test
    public void processGenericObject_GetMethodWithoutId_ReturnsMultipleObject() {
        createObject("objId1", "propValue1");
        createObject("objId2", "propValue2");

        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/process_generic_data/?http_method=GET&resource_path=my%2Ftype%2FName&__uid=" + uid.toString(),
                "");

        Assert.equals(200, response.getStatus());
        String expectedString =
                "\"result\":{\"items\":["
                + "{\"requiredProp\":\"propValue1\",\"key\":\"objId1\"},"
                + "{\"requiredProp\":\"propValue2\",\"key\":\"objId2\"}"
                + "]}";
        assertContains(response, expectedString);
    }

    @Test
    public void createGenericObjectTest() {
        String json = "{\"requiredProp\":\"value\"}";
        MockHttpServletResponse response = sendRequestUsual(
                "POST", "/generic_data/?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), json);

        Assert.equals(201, response.getStatus());

        assertContains(response, "\"requiredProp\":\"value\"");
        assertContainsRegexp(response, "\"key\":\"[\\w-]+\"");
    }

    @Test
    public void getGenericObject_ObjectAndTypeNameExist_ReturnsExpectedObject() {
        createObject("objId", "propValue");

        MockHttpServletResponse response = sendRequestUsual(
                "GET", "/generic_data/objId?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), "");

        Assert.equals(200, response.getStatus());
        assertContains(response, "\"result\":{\"requiredProp\":\"propValue\",\"key\":\"objId\"}");
        assertContainsInvocationInfo(response);
    }

    @Test
    public void getGenericObject_DirectSerialization() {
        createObject("objId", "propValue");

        MockHttpServletResponse response = sendRequestUsual(
                "GET", "/generic_data/objId?type_name=my%2Ftype%2FName&__uid=" + uid.toString() + "&direct=true", "");

        Assert.equals(200, response.getStatus());
        Assert.equals("{\"requiredProp\":\"propValue\",\"key\":\"objId\"}", getContent(response));
    }

    @Test
    public void getGenericObject_TypeNameDoesNotExist_Returns400() {
        MockHttpServletResponse response = sendRequestUsual(
                "GET", "/generic_data/objId?type_name=UnknownTypeName&__uid=" + uid.toString(), "");

        Assert.equals(400, response.getStatus());
    }

    @Test
    public void getGenericObjectsTest() {
        createObject("objId1", "propValue1");
        createObject("objId2", "propValue2");

        MockHttpServletResponse response = sendRequestUsual(
                "GET", "/generic_data/?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), "");

        Assert.equals(200, response.getStatus());
        String expectedString = "\"result\":{\"items\":["
                                + "{\"requiredProp\":\"propValue1\",\"key\":\"objId1\"},"
                                + "{\"requiredProp\":\"propValue2\",\"key\":\"objId2\"}"
                                + "]}";
        assertContains(response, expectedString);

        assertContainsInvocationInfo(response);
    }

    @Test
    public void getGenericObjectsTestNewUser() {

        MockHttpServletResponse response = sendRequestUsual(
                "GET", "/generic_data/?type_name=my%2Ftype%2FName&__uid=" + UserMetaManagerStub.WRONG_USER, "");

        Assert.equals(200, response.getStatus());
        String expectedString = "\"result\":{\"items\":[]}";
        assertContains(response, expectedString);

        assertContainsInvocationInfo(response);
    }

    @Test
    public void getGenericObjectsTest_WhereFilter() {
        createObject("obj1", "match");
        createObject("obj2", "mismatch");
        createObject("obj3", "match");

        Function<String, MockHttpServletResponse> request = value -> sendRequestUsual("GET",
                "/generic_data/?type_name=my%2Ftype%2FName&__uid=" + uid + "&where=" + UrlUtils.urlEncode(value), "");

        MockHttpServletResponse response = request.apply("requiredProp='match'");
        Assert.equals(200, response.getStatus());

        assertContains(response, "\"result\":{\"items\":["
                + "{\"requiredProp\":\"match\",\"key\":\"obj1\"},"
                + "{\"requiredProp\":\"match\",\"key\":\"obj3\"}"
                + "]}");

        response = request.apply("key='obj2'");
        Assert.equals(200, response.getStatus());

        assertContains(response, "\"result\":{\"items\":["
                + "{\"requiredProp\":\"mismatch\",\"key\":\"obj2\"}"
                + "]}");

        assertContainsInvocationInfo(response);
    }

    @Test
    public void getGenericObjectsTest_OrderBy() {
        createObject("obj1", "value3");
        createObject("obj2", "value1");
        createObject("obj3", "value2");

        Function<String, MockHttpServletResponse> request = value -> sendRequestUsual("GET",
                "/generic_data/?type_name=my%2Ftype%2FName&__uid=" + uid + "&order_by=" + UrlUtils.urlEncode(value), "");

        MockHttpServletResponse response = request.apply("requiredProp");
        Assert.equals(200, response.getStatus());

        assertContains(response, "\"result\":{\"items\":["
                + "{\"requiredProp\":\"value1\",\"key\":\"obj2\"},"
                + "{\"requiredProp\":\"value2\",\"key\":\"obj3\"},"
                + "{\"requiredProp\":\"value3\",\"key\":\"obj1\"}"
                + "]}");

        response = request.apply("-key");
        Assert.equals(200, response.getStatus());

        assertContains(response, "\"result\":{\"items\":["
                + "{\"requiredProp\":\"value2\",\"key\":\"obj3\"},"
                + "{\"requiredProp\":\"value1\",\"key\":\"obj2\"},"
                + "{\"requiredProp\":\"value3\",\"key\":\"obj1\"}"
                + "]}");
    }

    @Test
    public void getGenericObjectsTest_OrderedUuid() {
        ListF<String> ids = Cf.repeat(OrderedUUID::generateOrderedUUID, 3);
        ids.forEach(id -> createObject(id, id));

        Function3<String, String, Boolean, MockHttpServletResponse> request =
                (min, max, asc) -> sendRequestUsual("GET", "/generic_data/?type_name=my%2Ftype%2FName&__uid=" + uid
                        + Option.ofNullable(min).map(m -> "&query.range.from=" + m).mkString("")
                        + Option.ofNullable(max).map(m -> "&query.range.to=" + m).mkString("")
                        + Option.ofNullable(asc).map(m -> "&query.order.asc=" + m).mkString(""), "");

        Function3<String, String, Boolean, ListF<String>> requestRecords = (min, max, asc) -> {
            MockHttpServletResponse resp = request.apply(min, max, asc);

            Assert.equals(200, resp.getStatus());

            return Pattern2.compile("\"key\":\"([^\"]+)\"").findAllFirstGroups(getContent(resp));
        };

        Assert.equals(ids, requestRecords.apply(null, null, null));
        Assert.equals(ids, requestRecords.apply(null, null, true));
        Assert.equals(ids.reverse(), requestRecords.apply(null, null, false));

        Assert.equals(ids.drop(1), requestRecords.apply(ids.get(1), null, null));
        Assert.equals(ids.reverse().drop(1), requestRecords.apply(null, ids.get(1), false));

        Assert.equals(Cf.list(ids.get(1)), requestRecords.apply(ids.get(1), ids.get(1), true));
        Assert.equals(Cf.list(ids.get(1)), requestRecords.apply(ids.get(1), ids.get(1), false));

        Assert.equals(ids.take(2), requestRecords.apply(ids.get(0), ids.get(1), null));
        Assert.equals(ids.drop(1), requestRecords.apply(ids.get(1), ids.get(2), true));
        Assert.equals(ids.reverse().take(2), requestRecords.apply(ids.get(1), ids.get(2), false));
    }

    @Test
    public void getGenericObjectsTest_TotalCount() {
        ListF<String> ids = Cf.repeat(OrderedUUID::generateOrderedUUID, 3);
        ids.forEach(id -> createObject(id, id));

        Function2<Either<String, String>, Integer, MockHttpServletResponse> request =
                (uuidOrWhere, limit) -> sendRequestUsual("GET", "/generic_data/?type_name=my%2Ftype%2FName&__uid=" + uid
                        + Option.ofNullable(uuidOrWhere).map(e -> e.fold(
                                uuid -> "&query.range.from=" + uuid,
                                where -> "&where=" + UrlUtils.urlEncode(where))).mkString("")
                        + Option.ofNullable(limit).map(l -> "&limit=" + l).mkString(""), "");

        Function2<Either<String, String>, Integer, Option<Integer>> requestTotal = (uuidOrWhere, limit) -> {
            MockHttpServletResponse resp = request.apply(uuidOrWhere, limit);

            Assert.equals(200, resp.getStatus());

            return Pattern2.compile("\"total\":([^,]+)").findFirstGroup(getContent(resp)).map(Integer::parseInt);
        };

        Assert.none(requestTotal.apply(null, null));
        Assert.none(requestTotal.apply(Either.left(ids.first()), null));
        Assert.none(requestTotal.apply(Either.right("key>'0'"), null));

        Assert.some(3, requestTotal.apply(null, 2));
        Assert.some(3, requestTotal.apply(Either.left(ids.first()), 2));
        Assert.none(requestTotal.apply(Either.right("key>'0'"), 2));
    }

    @Test
    public void putGenericObject_WithoutRequiredProperty_MustThrowValidationError() {
        String json = "{\"key\":\"objId\"}";

        MockHttpServletResponse response = sendRequestUsual(
                "PUT", "/generic_data/objId?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), json);

        Assert.equals(400, response.getStatus());
        assertContains(response, "object has missing required properties ([\\\"requiredProp\\\"])");
    }

    @Test
    public void putGenericObject_WithNoKeyInPath_MustThrowMethodNotAllowed() {
        String json = "{\"requiredProp\":\"newPropValue\",\"key\":\"objId2\"}";

        MockHttpServletResponse response = sendRequestUsual(
                "PUT", "/generic_data/?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), json);

        Assert.equals(405, response.getStatus());
    }

    @Test
    public void putGenericObject_WithNewUser() {
        String json = "{\"requiredProp\":\"requiredPropValue\",\"key\":\"objId2\"}";

        MockHttpServletResponse response = sendRequestUsual(
                "PUT", "/generic_data/objId1?type_name=my%2Ftype%2FName&__uid=" + UserMetaManagerStub.WRONG_USER, json);

        Assert.equals(200, response.getStatus());
        assertContains(response, "{\"requiredProp\":\"requiredPropValue\",\"key\":\"objId1\"}");
    }

    @Test
    public void putGenericObject_WithKeyInBodyNotEqualsKeyInPath_MustIgnoreKeyInBody() {
        String json = "{\"requiredProp\":\"requiredPropValue\",\"key\":\"objId2\"}";

        MockHttpServletResponse response = sendRequestUsual(
                "PUT", "/generic_data/objId1?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), json);

        Assert.equals(200, response.getStatus());
        assertContains(response, "{\"requiredProp\":\"requiredPropValue\",\"key\":\"objId1\"}");
        Assert.notEmpty(getDataRecord("objId1"));
        Assert.isEmpty(getDataRecord("objId2"));
    }

    @Test
    public void putGenericObject_WithAdditionalProps_SaveIgnoringUnexpectedProps() {
        String json = "{\"requiredProp\":\"propValue\",\"key\":\"objId\", \"unknown\":\"unknownValue\"}";

        MockHttpServletResponse response = sendRequestUsual(
                "PUT", "/generic_data/objId?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), json);

        Assert.equals(200, response.getStatus());

        MapF<String, DataField> map = getDataFields("objId");

        Assert.sizeIs(1, map);
        Assert.equals("propValue", map.getTs("requiredProp").stringValue());
    }

    @Test
    public void putGenericObject_OldObjectExists_OverwritesOld() {
        createObject("objId", "oldPropValue");

        String json = "{\"requiredProp\":\"newPropValue\",\"key\":\"objId\"}";

        MockHttpServletResponse response = sendRequestUsual(
                "PUT", "/generic_data/objId?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), json);

        Assert.equals(200, response.getStatus());
        MapF<String, DataField> map = getDataFields("objId");

        Assert.equals("newPropValue", map.getTs("requiredProp").stringValue());
    }

    @Test
    public void putGenericObject_OverwritesIfMatch() {
        createObject("justInCase", "someValue");

        Function2<String, String, MockHttpServletResponse> put = (filter, json) ->
                sendRequestUsual("PUT", "/generic_data/objId?type_name=my%2Ftype%2FName&__uid=" + uid
                        + "&rewrite-if=" + UrlUtils.urlEncode(filter), json);

        Function0<String> loadValue = () -> getDataFields("objId").getOrThrow("requiredProp").stringValue();

        MockHttpServletResponse response = put.apply("requiredProp='missing'", "{\"requiredProp\":\"old\"}");
        Assert.equals(200, response.getStatus());

        Assert.equals("old", loadValue.apply());

        response = put.apply("requiredProp='mismatched'", "{\"requiredProp\":\"new\"}");
        Assert.equals(409, response.getStatus());

        Assert.equals("old", loadValue.apply());

        response = put.apply("requiredProp='old'", "{\"requiredProp\":\"new\"}");
        Assert.equals(200, response.getStatus());

        Assert.equals("new", loadValue.apply());

        response = put.apply("requiredProp!=$requiredProp", "{\"requiredProp\":\"new\"}");
        Assert.equals(409, response.getStatus());
    }

    private MapF<String, DataField> getDataFields(String objectId) {
        return getDataRecord(objectId).get().getData();
    }

    @Test
    public void deleteGenericObjectTest() {
        createObject("objId", "oldPropValue");
        MockHttpServletResponse response = sendRequestUsual(
                "DELETE", "/generic_data/objId?type_name=my%2Ftype%2FName&__uid=" + uid.toString(), "");

        Assert.equals(200, response.getStatus());
        Option<DataRecord> record = getDataRecord("objId");
        Assert.isEmpty(record);
    }

    private Option<DataRecord> getDataRecord(String objId) {
        return dataApiManager.getRecord(uid, typeSettings.toColRef().consRecordRef(objId));
    }

    private void createObject(String keyValue, String propValue) {
        MapF<String, DataField> dataFields = Cf.hashMap();
        dataFields.put("requiredProp", DataField.string(propValue));
        dataFields.put("key", DataField.string(keyValue));

        Database db = dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(uid, typeSettings.dbRef()));

        dataApiManager.applyDelta(
                db, RevisionCheckMode.PER_RECORD,
                new Delta(RecordChange.insert(typeSettings.typeLocation.collectionId, keyValue, dataFields)));
    }

    @Override
    protected String getNamespace() {
        return ru.yandex.chemodan.util.web.NS.API;
    }
}
