package ru.yandex.chemodan.app.dataapi.web.profile;

import java.util.List;

import lombok.Setter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.NS;
import ru.yandex.chemodan.app.dataapi.apps.profile.full.CachedFullProfileManager;
import ru.yandex.chemodan.app.dataapi.apps.profile.full.FullUserProfile;
import ru.yandex.chemodan.app.dataapi.web.DataApiActionBase;
import ru.yandex.chemodan.app.dataapi.web.auth.FromAttributeInterceptor;
import ru.yandex.chemodan.app.dataapi.web.auth.SetUserAndAppInfoInterceptorBase;
import ru.yandex.chemodan.http.proxy.ProxyManager;
import ru.yandex.chemodan.ratelimiter.yarl.YarlBaseInterceptor;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionDescriptor;
import ru.yandex.commune.a3.action.ActionDescriptorResolver;
import ru.yandex.commune.a3.action.CloneableActionDescriptor;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.http.ServletWebRequest;
import ru.yandex.commune.a3.action.intercept.ActionDispatcherInterceptor;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.Parameter;
import ru.yandex.commune.a3.action.parameter.ParameterResolver;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.result.error.ErrorResult;
import ru.yandex.commune.a3.action.result.error.ExceptionResolver;
import ru.yandex.commune.a3.action.result.json.JsonWriterActionResult;
import ru.yandex.commune.json.write.BenderJsonWriterWrapper;
import ru.yandex.commune.json.write.JsonWriter;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.serialize.BenderSerializer;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.net.uri.Uri2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.web.servlet.HttpServletResponseX;

/**
 * @author tolmalev
 */
@Action(
        value = @Action.Alias(value = "get-profile-batch", namespace = NS.PROFILE),
        description = "Выполнить batch запрос для получения данных про профиль пользователя")
@Path("/batch")
@WithMasterSlavePolicy(MasterSlavePolicy.R_ANY)
@Setter
public class ProfileBatchGetAction extends DataApiActionBase {

    private static final BenderSerializer<ErrorResult> errorRezultSerializer = Bender.serializer(ErrorResult.class);

    @RequestListParam(value = "actions")
    private ListF<String> queries;

    @SpecialParam
    private HttpServletRequestX req;
    @SpecialParam
    private HttpServletResponseX resp;

    private final CachedFullProfileManager profileManager;
    private final BenderMapper benderMapper = DataApiBenderUtils.mapper();
    private ActionDescriptorResolver actionDescriptorResolver;
    private ExceptionResolver exceptionResolver;
    private List<SetUserAndAppInfoInterceptorBase> setUserAndAppInfoInterceptors;

    public ProfileBatchGetAction(CachedFullProfileManager profileManager) {
        this.profileManager = profileManager;
    }


    public  ListF<ResultWithStatusCode> executeInner() throws Exception {
        req.setAttribute(FromAttributeInterceptor.ATTR_KEY, user);
        FullUserProfile profile = profileManager.getProfile(user);

        queries = queries
                .map(path -> StringUtils.removeStart(path, "/"))
                .map(path -> StringUtils.removeStart(path, "profile/"));

        ListF<ResultWithStatusCode> executionResults = callActions(profile, queries);

        return executionResults;
    }

    @Override
    public Object execute() throws Exception {
        ListF<ResultWithStatusCode> executionResults = executeInner();

        return (JsonWriterActionResult) (jw, additionalAttributes) -> {

            jw.writeObjectStart();
            writeInvocationInfo(jw, additionalAttributes);
            jw.writeArrayFieldStart("result");
            executionResults.forEach(result -> writeOneResult(jw, result));
            jw.writeArrayEnd();
            jw.writeObjectEnd();
        };
    }

    private void writeInvocationInfo(JsonWriter jw, Tuple2List<String, Object> additionalAttributes) {
        MapF<String, String> invocationInfo = additionalAttributes.map2(Cf.Object.toStringF()).toMap();

        jw.writeObjectFieldStart("invocationInfo");
        invocationInfo.entries().forEach(jw::writeStringField);
        jw.writeObjectEnd();
    }

    private void writeOneResult(JsonWriter jw, ResultWithStatusCode result) {
        jw.writeObjectStart();

        jw.writeNumberField("code", result.getStatusCode());

        if (result.result instanceof ErrorResult) {
            jw.writeFieldName("error");
            errorRezultSerializer.serializeJson((ErrorResult) result.result, new BenderJsonWriterWrapper(jw));
        } else {
            jw.writeFieldName("result");
            benderMapper.serializeJson(result.result, new BenderJsonWriterWrapper(jw));
        }

        jw.writeObjectEnd();
    }

    private ListF<ResultWithStatusCode> callActions(FullUserProfile profile, ListF<String> queries) {
        return queries.map(path -> {
            ServletWebRequest webRequest = new ServletWebRequest(req, resp);
            ActionInvocationContext context = new ActionInvocationContext(NS.PROFILE);
            context.setRequest(webRequest);

            Object result;

            try {
                Uri2 uri = Uri2.parse(path);

                for (Tuple2<String, String[]> entry : req.getParameterMap().entries()) {
                    webRequest.putParameter(entry._1, Cf.x(entry._2));
                }

                for (Tuple2<String, ListF<String>> t : uri.getQueryArgs().groupBy1().entries()) {
                    webRequest.putParameter(t._1, t._2);
                }

                webRequest.getHttpServletRequest().setAttribute(ProxyManager.SKIP_PROXY, "1");
                webRequest.getHttpServletRequest().setAttribute(YarlBaseInterceptor.RATE_LIMITER_IS_CHECKED, "1");
                for (ActionDispatcherInterceptor dispatcherInterceptor : setUserAndAppInfoInterceptors) {
                    dispatcherInterceptor.beforeDispatch(webRequest, context);
                }

                ActionDescriptor descriptor =
                        actionDescriptorResolver.resolveDescriptor(uri.getPath(), HttpMethod.GET, webRequest, context);

                if (descriptor.getContainerObject() instanceof GetProfilePartActionSupport) {
                    ListF<Parameter> parameters = ParameterResolver
                            .resolveParameters(descriptor.getParameterDescriptors(), webRequest, context);

                    GetProfilePartActionSupport action = (GetProfilePartActionSupport)
                            ObjectUtils.clone(((CloneableActionDescriptor) descriptor).getPrototype());

                    for (Parameter parameter : parameters) {
                        parameter.bindToProperty(action);
                    }

                    result = action.getFromFullProfile(profile);
                } else {
                    throw new IllegalArgumentException("Unsupported action called: " + path);
                }
            } catch (Exception e) {
                result = exceptionResolver.resolveException(webRequest, context, e);
            }
            return new ResultWithStatusCode(result, context.getHttpContext().getStatusCode());
        });
    }

    public void setActionDescriptorResolver(ActionDescriptorResolver actionDescriptorResolver) {
        this.actionDescriptorResolver = actionDescriptorResolver;
    }

    public void setExceptionResolver(ExceptionResolver exceptionResolver) {
        this.exceptionResolver = exceptionResolver;
    }

    public void setSetUserAndAppInfoInterceptors(List<SetUserAndAppInfoInterceptorBase> interceptors) {
        this.setUserAndAppInfoInterceptors = interceptors;
    }

    public static final class ResultWithStatusCode {
        public final Object result;
        public final Option<Integer> statusCode;

        private ResultWithStatusCode(Object result, Option<Integer> statusCode) {
            this.result = result;
            this.statusCode = statusCode;
        }

        public int getStatusCode() {
            if (statusCode.isPresent()) {
                return statusCode.get();
            } else {
                if (result instanceof ErrorResult) {
                    return 500;
                } else {
                    return 200;
                }
            }
        }
    }

    public int getRequestCount() {
        return Option.ofNullable(queries)
                .map(List::size)
                .orElse(1);
    }
}
