package ru.yandex.chemodan.app.dataapi.web.profile.yatickets.orders;

import org.junit.Test;

import ru.yandex.chemodan.app.dataapi.NS;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.web.test.ApiTestBase;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class OrdersApiTest extends ApiTestBase {
    private static final String ORDER_FULL_DATA_FILE = "OrderFullData.json";
    private static final String ORDER_MIN_DATA_FILE = "OrderMinData.json";
    public static final String MIN_ORDER_ID = "minOrderId";

    @Test
    public void createNewOrder_OrderWithFullData_Returns201() {
        createOrderAndCheckResult(ORDER_FULL_DATA_FILE, 201);
    }

    @Test
    public void createNewOrder_OrderWithMinData_Returns201() {
        createOrderAndCheckResult(ORDER_MIN_DATA_FILE, 201);
    }

    @Test
    public void createNewOrder_ExistedOrder_Returns400() {
        createOrderAndCheckResult(ORDER_MIN_DATA_FILE, 201);
        createOrderAndCheckResult(ORDER_MIN_DATA_FILE, 400);
    }

    @Test
    public void setOrder_OldOrderNotExists_Returns201() {
        setOrderAndCheckResult(ORDER_MIN_DATA_FILE, MIN_ORDER_ID, 201);
    }

    @Test
    public void setOrder_OldOrderExists_Returns200() {
        setOrderAndCheckResult(ORDER_MIN_DATA_FILE, MIN_ORDER_ID, 201);
        setOrderAndCheckResult(ORDER_MIN_DATA_FILE, MIN_ORDER_ID, 200);
    }

    @Test
    public void setOrder_OrderIdInPathNotEqualsWithOrderIdInRequest_Returns400() {
        setOrderAndCheckResult(ORDER_MIN_DATA_FILE, "wrongOrderId", 400);
    }

    @Test
    public void getOrder_ExistedOrder_ReturnsOrderAnd200() throws Exception {
        MockHttpServletResponse response = createOrderAndCheckResult(ORDER_FULL_DATA_FILE, 201);
        String resultedJson = getOrderAndCheckResult("fullOrderId", 200);

        String fileData = getOrderData(ORDER_FULL_DATA_FILE);
        Assert.isTrue(resultedJson.contains(fileData));

        assertContainsInvocationInfo(response);
    }

    @Test
    public void getOrder_NotExistedOrder_Returns404() throws Exception {
        getOrderAndCheckResult("unknownOrderId", 404);
    }

    @Test
    public void getAllOrders_ReturnsAllOrdersAnd200() throws Exception {
        createOrderAndCheckResult(ORDER_FULL_DATA_FILE, 201);
        createOrderAndCheckResult(ORDER_MIN_DATA_FILE, 201);

        String url = StringUtils.format("/ya-tickets/orders?__uid={}", uid.toString());
        MockHttpServletResponse response = sendRequestUsual("GET", url, "");
        Assert.equals(200, response.getStatus());

        String resultedJson = response.getContentAsString();

        String fullOrderData = getOrderData(ORDER_FULL_DATA_FILE);
        String minOrderData = getOrderData(ORDER_MIN_DATA_FILE);
        Assert.isTrue(resultedJson.contains(fullOrderData));
        Assert.isTrue(resultedJson.contains(minOrderData));

        assertContainsInvocationInfo(response);
    }

    @Test
    public void deleteOrder_ExistedOrder_Returns200() {
        createOrderAndCheckResult(ORDER_MIN_DATA_FILE, 201);
        deleteOrderAndCheckResult(MIN_ORDER_ID, 200);
    }

    @Test
    public void deleteOrder_NotExistedOrder_Returns404() {
        deleteOrderAndCheckResult(MIN_ORDER_ID, 404);
    }

    private String getOrderAndCheckResult(String orderId, int expectedResult) throws Exception {
        String url = getSpecificOrderUrl(orderId);
        MockHttpServletResponse response = sendRequestUsual("GET", url, "");
        Assert.equals(expectedResult, response.getStatus());
        return response.getContentAsString();
    }

    private MockHttpServletResponse createOrderAndCheckResult(String jsonFile, int expectedResult) {
        String data = getOrderData(jsonFile);

        MockHttpServletResponse response = sendRequestUsual(
                "POST",
                "/ya-tickets/orders?__uid=" + uid.toString(),
                data);

        Assert.equals(expectedResult, response.getStatus());
        return response;
    }

    private void deleteOrderAndCheckResult(String orderId, int expectedResult) {
        MockHttpServletResponse response = sendRequestUsual("DELETE", getSpecificOrderUrl(orderId), "");

        Assert.equals(expectedResult, response.getStatus());
    }

    private String getSpecificOrderUrl(String orderId) {
        return StringUtils.format("/ya-tickets/orders/{}?__uid={}", orderId, uid.toString());
    }

    private void setOrderAndCheckResult(String jsonFile, String orderId, int expectedResult) {
        String data = getOrderData(jsonFile);
        String url = getSpecificOrderUrl(orderId);
        MockHttpServletResponse response = sendRequestUsual("PUT", url, data);

        Assert.equals(expectedResult, response.getStatus());
    }

    private String getOrderData(String jsonFile) {
        ClassPathResourceInputStreamSource iss = new ClassPathResourceInputStreamSource(
                getClass(), jsonFile);
        return iss.readText().replaceAll(" |\\n", "");
    }

    @Override
    protected String getNamespace() {
        return NS.PROFILE;
    }
}
