package ru.yandex.chemodan.app.dataapi.web.ratelimiter;

import java.util.function.Supplier;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.web.batch.BatchAllAction;
import ru.yandex.chemodan.app.dataapi.web.batch.BatchGetAction;
import ru.yandex.chemodan.app.dataapi.web.profile.ProfileBatchGetAction;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.log.DiskLog4jRequestLog;
import ru.yandex.chemodan.ratelimiter.yarl.YarlHttpClient;
import ru.yandex.chemodan.ratelimiter.yarl.YarlWithClientFilterInterceptor;
import ru.yandex.commune.a3.action.CloneableAction;
import ru.yandex.commune.a3.action.invoke.ActionInvocation;
import ru.yandex.commune.a3.action.invoke.CloneableActionInvocation;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author friendlyevil
 */
public class DataapiRateLimiterInterceptor extends YarlWithClientFilterInterceptor {
    private final AliveAppInfo aliveAppInfo;
    private final Supplier<Boolean> perDcEnabled;
    private final Supplier<Boolean> perDcForAllClients;
    private final Supplier<ListF<String>> perDcForClients;

    public DataapiRateLimiterInterceptor(YarlHttpClient yarlHttpClient,
                                         Supplier<Boolean> isDryRun,
                                         Supplier<Boolean> isEnabled,
                                         Supplier<Boolean> isWhiteListMode,
                                         Supplier<ListF<String>> enabledForClients,
                                         Supplier<ListF<String>> disabledForClients,
                                         AliveAppInfo aliveAppInfo,
                                         Supplier<Boolean> perDcEnabled,
                                         Supplier<Boolean> perDcForAllClients,
                                         Supplier<ListF<String>> perDcForClients) {
        super(yarlHttpClient, isDryRun, isEnabled, isWhiteListMode, enabledForClients, disabledForClients);
        this.aliveAppInfo = aliveAppInfo;
        this.perDcEnabled = perDcEnabled;
        this.perDcForAllClients = perDcForAllClients;
        this.perDcForClients = perDcForClients;
    }

    @Override
    protected boolean needRateLimitCheck(HttpServletRequestX request) {
        return super.needRateLimitCheck(request) && !isRestRequest();
    }

    protected boolean isRestRequest() {
        return YandexCloudRequestIdHolder.getO().map(ycrid -> ycrid.startsWith("rest")).orElse(false);
    }

    @Override
    protected int getWeight(ActionInvocation invocation) {
        if (invocation instanceof CloneableActionInvocation) {
            CloneableActionInvocation inv = (CloneableActionInvocation) invocation;
            CloneableAction action = inv.getAction();

            if (action instanceof BatchAllAction) {
                return ((BatchAllAction) action).getRequestCount() + 1;
            } else if (action instanceof BatchGetAction) {
                return ((BatchGetAction) action).getRequestCount(inv.getContext()) + 1;
            } else if (action instanceof ProfileBatchGetAction) {
                return ((ProfileBatchGetAction) action).getRequestCount() + 1;
            }
        }

        return 1;
    }

    @Override
    protected String clientPrefix() {
        return "dataapi_";
    }

    @Override
    protected Option<String> getClientId(HttpServletRequestX request) {
        return getClientIdFromRequest(request).map(this::appendDcIfNeed);
    }

    private String appendDcIfNeed(String clientId) {
        Option<String> dc = aliveAppInfo.getDc();
        if (dc.isPresent() && perDcEnabled.get()
                && (perDcForAllClients.get() || perDcForClients.get().containsTs(clientId))) {
            return clientId + '_' + dc.get();
        }

        return clientId;
    }

    private Option<String> getClientIdFromRequest(HttpServletRequestX request) {
        return Option.ofNullable(request.getAttribute(DiskLog4jRequestLog.CLIENT_ID_ATTRIBUTE))
                .map(Object::toString);
    }
}
