package ru.yandex.chemodan.app.dataapi.web.ratelimiter;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.util.ReflectionTestUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.apps.profile.address.Address;
import ru.yandex.chemodan.app.dataapi.apps.profile.address.AddressManager;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.web.test.ApiTestBase;
import ru.yandex.chemodan.cloud.auth.config.PlatformClient;
import ru.yandex.chemodan.cloud.auth.config.ZkPlatformClientRepository;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.ratelimiter.yarl.RateLimiterStatus;
import ru.yandex.chemodan.ratelimiter.yarl.YarlBaseInterceptor;
import ru.yandex.chemodan.ratelimiter.yarl.YarlHttpClient;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

import static org.junit.Assert.assertEquals;

/**
 * @author friendlyevil
 */
@ActivateDataApiEmbeddedPg
public class RateLimiterTest extends ApiTestBase {

    @Autowired
    private ZkPlatformClientRepository repository;

    @Autowired
    private AddressManager addressManager;

    @Autowired
    private DataapiRateLimiterInterceptor rateLimiterInterceptor;

    private YarlHttpClient yarlHttpClient;

    @Before
    public void before() {
        super.before();
        yarlHttpClient = Mockito.mock(YarlHttpClient.class);
        ReflectionTestUtils.setField(rateLimiterInterceptor, "yarlHttpClient", yarlHttpClient);
        ReflectionTestUtils.setField(rateLimiterInterceptor, "isWhiteListMode", (Supplier<Boolean>) () -> false);
        ReflectionTestUtils.setField(rateLimiterInterceptor, "isDryRun", (Supplier<Boolean>) () -> false);
        ReflectionTestUtils.setField(rateLimiterInterceptor, "isEnabled", (Supplier<Boolean>) () -> true);
        ReflectionTestUtils.setField(rateLimiterInterceptor, "perDcEnabled", (Supplier<Boolean>) () -> false);
    }

    private void createClientWithClientToken() {
        PlatformClient platformClient = new PlatformClient();
        platformClient.setEnabled(true);
        platformClient.setOauthClientId(Option.of("morda"));
        platformClient.setOauthScopes(Cf.arrayList("cloud_api.profile:generic.addresses.read",
                "cloud_api.profile:generic.addresses.write"));
        repository.setClientsByToken(Cf.map("c14e9620915645ba832f3a09264067e2", platformClient));
    }

    private MockHttpServletResponse sendRequestWithClientToken(String action, String url, String content) {
        return sendRequest(action, "", url, content,
                Cf.map("Authorization", "ClientToken token=c14e9620915645ba832f3a09264067e2;uid=" + uid.toString() +
                        ";"));
    }

    @Test
    public void testCallYarlWithClientTokenAuthorization() {
        doAddressRequestWithClientTokenAuthorization(200, 1);
    }

    @Test
    public void dontCallRateLimiterForRequestWithHeader() {
        addressManager.createAddress(uid, new Address("addressId"));
        createClientWithClientToken();

        AtomicInteger count = mockYarl("dataapi_morda", 1, true);

        MockHttpServletResponse response = sendRequest("GET", "",
                "/platform/personality/profile/addresses/addressId", "",
                Cf.map("Authorization", "ClientToken token=c14e9620915645ba832f3a09264067e2;uid=" + uid.toString() + ";",
                        YarlBaseInterceptor.RATE_LIMITER_IS_CHECKED, "1"));

        assertEquals(response.getStatus(), 200);
        assertEquals(count.get(), 0);
    }

    @Test
    public void testLimitClientWithTokenAuthorization() {
        addressManager.createAddress(uid, new Address("addressId"));
        createClientWithClientToken();

        AtomicInteger count = mockYarl("dataapi_morda", 1, false);

        MockHttpServletResponse response = sendRequestWithClientToken("GET",
                "/platform/personality/profile/addresses/addressId", "");

        assertEquals(response.getStatus(), 429);
        assertEquals(count.get(), 1);
    }

    @Test
    public void testRateLimiterForBatchRequest() {
        createClientWithClientToken();
        AtomicInteger count = mockYarl("dataapi_morda", 3, true);

        MockHttpServletResponse response = sendRequestWithClientToken(
                "POST", "/platform/batch/?uid=123", "{"
                        + "\"items\" : ["
                        + "{\"body\":\"{\\\"title\\\":\\\"Дом\\\",\\\"data_key\\\":\\\"home\\\",\\\"latitude\\\":55" +
                        ".7275871665287,\\\"address_line\\\":\\\"Россия, Москва, 4-й Какой-то переулок, 4\\\"," +
                        "\\\"longitude\\\":37.6183377919913,\\\"tags\\\":\\\"_w-_traffic-1-start\\\"," +
                        "\\\"address_line_short\\\":\\\"4-й Какой-то, 4\\\"}\",\"method\":\"PUT\"," +
                        "\"relative_url\":\"/v1/personality/profile/addresses/\"}," +
                        "{\"relative_url\":\"/v1/personality/profile/addresses/\",\"method\":\"PUT\"," +
                        "\"body\":\"{\\\"longitude\\\":37.588144,\\\"tags\\\":\\\"_w-_traffic-1-finish\\\"," +
                        "\\\"address_line_short\\\":\\\"улица Льва Толстого, 16\\\",\\\"title\\\":\\\"Работа\\\"," +
                        "\\\"data_key\\\":\\\"work\\\",\\\"address_line\\\":\\\"Россия, Москва, улица Льва Толстого, " +
                        "16\\\",\\\"latitude\\\":55.733842}\"}"
                        + "]"
                        + "}");

        assertContains(response, "href");
        assertEquals(count.get(), 1);
    }

    @Test
    public void dontCallRateLimiterForRestReqeuest() {
        YandexCloudRequestIdHolder.set("rest-b9a975159648713bc42397d81f46fe16-api50h");
        doAddressRequestWithClientTokenAuthorization(200, 0);
    }

    @Test
    public void testBlackList() {
        ReflectionTestUtils.setField(rateLimiterInterceptor, "disabledForClients",
                (Supplier<ListF<String>>) () -> Cf.list("morda"));
        doAddressRequestWithClientTokenAuthorization(200, 0);
    }

    @Test
    public void testWhiteListDontContainClient() {
        ReflectionTestUtils.setField(rateLimiterInterceptor, "isWhiteListMode", (Supplier<Boolean>) () -> true);
        ReflectionTestUtils.setField(rateLimiterInterceptor, "enabledForClients", (Supplier<ListF<String>>) Cf::list);
        doAddressRequestWithClientTokenAuthorization(200, 0);
    }

    @Test
    public void testLimitClientByPerDcCounter() {
        ReflectionTestUtils.setField(rateLimiterInterceptor, "perDcEnabled", (Supplier<Boolean>) () -> true);
        ReflectionTestUtils.setField(rateLimiterInterceptor, "perDcForAllClients", (Supplier<Boolean>) () -> true);
        ReflectionTestUtils.setField(rateLimiterInterceptor, "aliveAppInfo",
                new AliveAppInfo("serviceName", "appName", new Instant(), "", "host", 1, "", Option.of("sas")));
        addressManager.createAddress(uid, new Address("addressId"));
        createClientWithClientToken();

        AtomicInteger count = mockYarl("dataapi_morda_sas", 1, true);

        MockHttpServletResponse response = sendRequestWithClientToken("GET",
                "/platform/personality/profile/addresses/addressId", "");

        assertEquals(response.getStatus(), 200);
        assertEquals(count.get(), 1);
    }

    private void doAddressRequestWithClientTokenAuthorization(int status, int rlCount) {
        addressManager.createAddress(uid, new Address("addressId"));
        createClientWithClientToken();

        AtomicInteger count = mockYarl("dataapi_morda", 1, true);

        MockHttpServletResponse response = sendRequestWithClientToken("GET",
                "/platform/personality/profile/addresses/addressId", "");

        assertEquals(response.getStatus(), status);
        assertEquals(count.get(), rlCount);
    }

    @Override
    protected String getNamespace() {
        return ru.yandex.chemodan.util.web.NS.API;
    }

    private AtomicInteger mockYarl(String service, int weight, boolean accept) {
        AtomicInteger invocationCount = new AtomicInteger(0);
        Mockito.when(yarlHttpClient.checkLimit(Mockito.anyString(), Mockito.anyInt())).then(invocation -> {
            invocationCount.incrementAndGet();
            assertEquals(invocation.getArgument(0), service);
            assertEquals((int) invocation.getArgument(1), weight);

            return accept ? RateLimiterStatus.ACCEPTED : RateLimiterStatus.REJECTED;
        });

        return invocationCount;
    }
}
