package ru.yandex.chemodan.app.dataapi.web.rest;

import org.junit.Test;
import org.springframework.context.annotation.Bean;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.dataapi.web.DataApiActionsContextConfiguration;
import ru.yandex.chemodan.app.dataapi.web.DeleteDatabaseAction;
import ru.yandex.chemodan.app.dataapi.web.DeleteDatabasesAction;
import ru.yandex.chemodan.app.dataapi.web.GetDatabaseAction;
import ru.yandex.chemodan.app.dataapi.web.GetOrCreateDatabaseAction;
import ru.yandex.chemodan.app.dataapi.web.GetSnapshotAction;
import ru.yandex.chemodan.app.dataapi.web.ListDatabasesAction;
import ru.yandex.chemodan.app.dataapi.web.ListDeltasAction;
import ru.yandex.chemodan.app.dataapi.web.PatchDatabaseAction;
import ru.yandex.chemodan.app.dataapi.web.PutDeltaAction;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.RegexUriPattern;
import ru.yandex.commune.a3.action.UriMatcher;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.MethodX;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class ActionsPathTest {

    @Test
    public void getDatabases() {
        checkRestAndUsual(ListDatabasesAction.class,
                HttpMethod.GET, "/databases/",
                Cf.map());
    }

    @Test
    public void getDatabase() {
        checkRestAndUsual(GetDatabaseAction.class,
                HttpMethod.GET, "/databases/123",
                Cf.map(
                        "databaseId", "123"
                ));
    }

    @Test
    public void getOrCreateDatabase() {
        checkRestAndUsual(GetOrCreateDatabaseAction.class,
                HttpMethod.PUT, "/databases/123",
                Cf.map(
                        "databaseId", "123"
                ));
    }

    @Test
    public void deleteDatabase() {
        checkRestAndUsual(DeleteDatabaseAction.class,
                HttpMethod.DELETE, "/databases/123",
                Cf.map(
                        "databaseId", "123"
                ));
    }

    @Test
    public void patchDatabase() {
        checkRestAndUsual(PatchDatabaseAction.class,
                HttpMethod.PATCH, "/databases/123",
                Cf.map(
                        "databaseId", "123"
                ));
    }

    @Test
    public void deleteDatabases() {
        checkRestAndUsual(DeleteDatabasesAction.class,
                HttpMethod.DELETE, "/databases/",
                Cf.map());
    }

    @Test
    public void getSnapshot() {
        checkRestAndUsual(GetSnapshotAction.class,
                HttpMethod.GET, "/databases/123/snapshot",
                Cf.map(
                        "databaseId", "123"
                ));
    }

    @Test
    public void putDelta() {
        checkRestAndUsual(PutDeltaAction.class,
                HttpMethod.PUT, "/databases/123/deltas",
                Cf.map(
                        "databaseId", "123"
                ));

        checkRestAndUsual(PutDeltaAction.class,
                HttpMethod.PUT, "/databases/123/diffs",
                Cf.map(
                        "databaseId", "123"
                ));
    }

    @Test
    public void getDeltas() {
        checkRestAndUsual(ListDeltasAction.class,
                HttpMethod.GET, "/databases/123/diffs",
                Cf.map(
                        "databaseId", "123"
                ));

        checkRestAndUsual(ListDeltasAction.class,
                HttpMethod.GET, "/databases/123/deltas",
                Cf.map(
                        "databaseId", "123"
                ));
    }

    private static ListF<ClassX<Object>> actionClasses =
            Cf.x(DataApiActionsContextConfiguration.class.getDeclaredMethods())
                .map(m -> MethodX.wrap(m))
                .filter(m -> m.hasAnnotation(Bean.class))
                .filter(m -> m.getReturnType().hasAnnotation(Path.class))
                .map(m -> m.getReturnType());

    private void checkRestAndUsual(
            Class actionClass,
            HttpMethod method,
            String url,
            MapF<String, String> expected)
    {
        check(actionClass, method, url, expected.plus1("context", ""));
        check(actionClass, method, "/app" + url, expected.plus1("context", "app/"));
        check(actionClass, method, "/user" + url, expected.plus1("context", "user/"));
    }

    private void check(
            Class actionClass,
            HttpMethod method,
            String url,
            MapF<String, String> expected)
    {
        ListF<ClassX<Object>> matchClasses = actionClasses.filter(clazz -> matches(clazz, url, method));

        String message = method + " " + url + " must match only " + actionClass + " but matches " + matchClasses;
        Assert.equals(1, matchClasses.size(), message);

        Assert.equals(ClassX.wrap(actionClass), matchClasses.single(), message );

        String pattern = ((Path) actionClass.getAnnotation(Path.class)).value();

        checkOne(pattern, url, expected);
        checkOne(pattern, StringUtils.removeStart(url, "/"), expected);
        checkOne(pattern, StringUtils.removeEnd(url, "/"), expected);
        checkOne(pattern, StringUtils.removeStart(StringUtils.removeEnd(url, "/"), "/"), expected);
    }

    private void checkOne(
            String pattern,
            String url,
            MapF<String, String> expected)
    {
        RegexUriPattern regex = RegexUriPattern.compile(pattern);
        UriMatcher matcher = regex.matcher(url);

        Assert.isTrue(matcher.matches(), url + " not matches " + pattern);
        MapF<String, String> params = matcher.getParams(regex.getNamedGroups());

        expected.entries().forEach((key, value) -> Assert.some(value, params.getO(key)));
    }

    private boolean matches(ClassX actionClass, String url, HttpMethod method) {
        String pattern = ((Path) actionClass.getAnnotation(Path.class)).value();
        ListF<HttpMethod> methods = Cf.x(((Path) actionClass.getAnnotation(Path.class)).methods());

        return methods.containsTs(method) && (
                RegexUriPattern.compile(pattern).matcher(url).matches()
                || RegexUriPattern.compile(pattern).matcher(StringUtils.removeStart(url, "/")).matches()
        );
    }
}
