package ru.yandex.chemodan.app.dataapi.web.rest;

import java.io.UnsupportedEncodingException;

import com.fasterxml.jackson.databind.JsonNode;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseAccessType;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.web.test.ApiTestBase;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.chemodan.util.tvm.TvmClientInfo;
import ru.yandex.chemodan.util.tvm.TvmClientInfoRegistry;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.spring.ServicesStarter;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

/**
 * @author tolmalev
 */
@ContextConfiguration(classes = {
        RestLiteDatabaseApiTest.Context.class,
})
@ActivateDataApiEmbeddedPg
public class RestLiteDatabaseApiTest extends ApiTestBase {

    @Autowired
    private DataApiManager dataApiManager;

    @Autowired
    private ServicesStarter servicesStarter;

    @Autowired
    private TvmClientInfoRegistry tvmClientInfoRegistry;

    @After
    public void after() {
        timeStart();
    }

    @Before
    public void before() {
        super.before();
        try {
            servicesStarter.start();
        } catch (RuntimeException ignored) {}

        tvmClientInfoRegistry.put(new TvmClientInfo(
                67, "cloud-api", true, true, "REST for tests"
        ));
    }

    @Override
    protected String getNamespace() {
        return ru.yandex.chemodan.util.web.NS.API;
    }

    @Test
    public void getNotExistingDb() {
        testForAppDb("GET", "h_app", (context, app, dbId, resp) -> {
            Assert.equals(404, resp.getStatus());
            // TODO: validate error format
            JsonNode node = JsonNodeUtils.getNode(resp.getContentAsString());

            Assert.isTrue(node.has("invocationInfo"));
        }, "dbId");
    }

    @Test
    public void getNotExistingDbWithFields() {
        testForAppDb("GET", "h_app", Cf.map("fields", "handle"),(context, app, dbId, resp) -> {
            Assert.equals(404, resp.getStatus());

            JsonNode node = JsonNodeUtils.getNode(resp.getContentAsString());

            Assert.isTrue(node.has("invocationInfo"));
        }, "dbId");
    }

    @Test
    public void listDatabases() throws UnsupportedEncodingException {
        createDb("dbId_1");
        createDb("dbId_2");
        createDb("dbId_3");

        MockHttpServletResponse resp = sendRequestRestLike("GET", "/app/databases/?limit=10&__uid=" + uid + "&app=h_app", "");

        Assert.equals(200, resp.getStatus());

        JsonNode node = JsonNodeUtils.getNode(resp.getContentAsString());

        Assert.equals(0, node.get("offset").asInt());
        Assert.equals(3, node.get("total").asInt());
        Assert.equals(10, node.get("limit").asInt());

        Assert.isTrue(node.get("items").isArray());
        Assert.equals(node.get("total").asInt(), node.get("items").size());

        ListF<JsonNode> sortedDbNodes = Cf.x(node.get("items").elements()).toList().sortedBy(n -> n.get("dbId"));

        for (int i = 1; i < node.get("total").asInt() + 1; i++) {
            validateDbFormat(sortedDbNodes.get(i - 1), "dbId_" + i);
        }
    }

    @Test
    public void listDatabasesWithFields() throws UnsupportedEncodingException {
        createDb("dbId_1");
        createDb("dbId_2");
        createDb("dbId_3");

        MockHttpServletResponse resp = sendRequestRestLike("GET", "/app/databases/?limit=10&fields=total,offset,items.modified,items.handle,items.database_id&__uid=" + uid + "&app=h_app", "");

        Assert.equals(200, resp.getStatus());

        JsonNode node = JsonNodeUtils.getNode(resp.getContentAsString());

        Assert.equals(0, node.get("offset").asInt());
        Assert.equals(3, node.get("total").asInt());
        Assert.isFalse(node.has("limit"));

        Assert.isTrue(node.get("items").isArray());
        Assert.equals(node.get("total").asInt(), node.get("items").size());

        for (int i = 1; i < node.get("total").asInt() + 1; i++) {
            JsonNode db = node.get("items").get(i - 1);
            String dbId = "dbId_" + i;

            Assert.isTrue(db.get("handle").isTextual());

            Assert.isFalse(db.has("records_count"));
            Assert.isFalse(db.has("revision"));
            Assert.isFalse(db.has("size"));

            Assert.isTrue(db.get("database_id").isTextual());
            Assert.equals(dbId, db.get("database_id").asText());

            Assert.isFalse(db.has("created"));
            validateDateFormat(db.get("modified"));
        }
    }

    @Test
    public void getOrCreateDatabase_create() {
        // we delete DB to get 201 at the next request too
        dataApiManager.deleteDatabaseIfExists(
                new UserDatabaseSpec(uid, DatabaseRef.cons(Option.of("h_app"), "dbId")),
                DatabaseDeletionMode.REMOVE_COMPLETELY
        );

        // we delete DB to get 201 at the next request too
        dataApiManager.deleteDatabaseIfExists(
                new UserDatabaseSpec(uid, DatabaseRef.cons(Option.empty(), "dbId")),
                DatabaseDeletionMode.REMOVE_COMPLETELY
        );

        testForAppDb("PUT", "h_app", (context, app, dbId, resp) -> {
            Assert.equals(201, resp.getStatus());
            validateDbFormat(resp.getContentAsString(), dbId);

            // we delete DB to get 201 at the next request too
            dataApiManager.deleteDatabaseIfExists(
                    new UserDatabaseSpec(uid, DatabaseRef.cons(Option.of("h_app"), "dbId")),
                    DatabaseDeletionMode.REMOVE_COMPLETELY
            );

            // we delete DB to get 201 at the next request too
            dataApiManager.deleteDatabaseIfExists(
                    new UserDatabaseSpec(uid, DatabaseRef.cons(Option.empty(), "dbId")),
                    DatabaseDeletionMode.REMOVE_COMPLETELY
            );
        }, "dbId");
    }

    @Test
    public void getOrCreateDatabase_get() {
        createDb();
        createDb("dbId", "user");

        testForAppDb("PUT", "h_app", (context, app, dbId, resp) -> {
            Assert.equals(200, resp.getStatus());
            validateDbFormat(resp.getContentAsString(), dbId);
        }, "dbId");
    }

    @Test
    public void getExistingDb() {
        createDb();
        createDb("dbId", "user");

        testForAppDb("GET", "h_app", (context, app, dbId, resp) -> {
            Assert.equals(200, resp.getStatus());
            validateDbFormat(resp.getContentAsString(), dbId);
        }, "dbId");
    }

    private void testForAppDb(String method, String app, DbRespValidator validator, String dbId) {
        testForAppDb(method, app, Cf.map(), validator, dbId);
    }

    private void testForAppDb(String method, String app, MapF<String, String> params, DbRespValidator validator,
            String dbId)
    {
        testForDbInt(method, dbId, app, "app", params, validator);
        testForDbInt(method, dbId, app, "user", params, validator);
        testForDbInt(method, ".ext." + app + "@" + dbId, "client_app", "app", params, validator);
    }

    interface DbRespValidator {
        void validate(String context, String app, String dbId, MockHttpServletResponse response) throws Exception;
    }

    private void testForDbInt(String method, String dbId, String app, String context,
            MapF<String, String> params, DbRespValidator validator)
    {
        String url = UriBuilder.cons()
                .appendPath("/" + context + "/databases/")
                .appendPath(dbId)
                .addParam("__uid", uid)
                .addParam("app", app)
                .addParams(params.entries())
                .build()
                .toString();

        MockHttpServletResponse resp = sendRequestRestLike(method, url, "");
        try {
            validator.validate(context, app, dbId, resp);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }

        resp = sendRequestRestLike(method, url, "", Cf.map());
        try {
            validator.validate(context, app, dbId, resp);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }

//        url = UriBuilder.cons()
//                .appendPath("/databases/")
//                .appendPath(dbId)
//                .addParam("__uid", uid)
//                .addParam("context", context)
//                .addParam("app", app)
//                .addParam("direct", true)
//                .addParams(params.entries())
//                .build()
//                .toString();
//
//        resp = sendRequestUsual(method, url, "");
//        try {
//            validator.validate(app, dbId, resp);
//        } catch (Exception e) {
//            throw ExceptionUtils.translate(e);
//        }
//
//        resp = sendRequestUsual(method, url, "", Cf.map("Ticket", restBackendTicketHolder.getTicket()));
//        try {
//            validator.validate(app, dbId, resp);
//        } catch (Exception e) {
//            throw ExceptionUtils.translate(e);
//        }
    }

    private void createDb() {
        createDb("dbId");
    }

    private void validateDbFormat(String resp, String dbId) {
        validateDbFormat(JsonNodeUtils.getNode(resp), dbId);
    }

    private void validateDbFormat(JsonNode db, String dbId) {
        Assert.isTrue(db.get("handle").isTextual());

        Assert.isTrue(db.get("records_count").isInt());
        Assert.isTrue(db.get("revision").isInt());
        Assert.isTrue(db.get("size").isInt());

        Assert.isTrue(db.get("database_id").isTextual());
        Assert.equals(dbId, db.get("database_id").asText());

        validateDateFormat(db.get("created"));
        validateDateFormat(db.get("modified"));
    }

    private void validateDateFormat(JsonNode node) {
        Assert.isTrue(node.isTextual());
        validateDateFormat(node.asText());
    }

    private void validateDateFormat(String value) {
        // 2014-10-28T14:54:20.118000+00:00
        Assert.isTrue(Pattern2.compile(
                "\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}\\.\\d{6}\\+\\d{2}:\\d{2}"
        ).matches(value), "Bad date format: " + value);
    }

    private void createDb(String dbId) {
        createDb(dbId, "app");
    }

    private void createDb(String dbId, String context) {
        MockHttpServletResponse r1 = sendRequestRestLike(
                "PUT",
                "/" + context + "/databases/" + dbId + "?__uid=" + uid + "&app=h_app",
                ""
        );
        Assert.equals(201, r1.getStatus());
    }

    @ContextConfiguration
    static class Context {
        @Bean
        @Primary
        public ExternalDatabasesRegistry externalDatabasesRegistry() {
            ExternalDatabasesRegistry mock = Mockito.mock(ExternalDatabasesRegistry.class);
            Mockito
                    .when(mock.getExternalDatabaseAccessType(Mockito.any()))
                    .then(invocation -> {
                ExternalDatabaseAlias alias = (ExternalDatabaseAlias) invocation.getArguments()[0];
                if (alias.clientAppName().equals("client_app")) {
                    return Option.of(DatabaseAccessType.READ_WRITE);
                }
                if (alias.clientAppName().equals("ro_client_app")) {
                    return Option.of(DatabaseAccessType.READ_ONLY);
                }
                return Option.empty();
            });
            return mock;
        }
    }
}
