package ru.yandex.chemodan.app.dataapi.web.xiva;

import com.fasterxml.jackson.databind.JsonNode;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiYaTeamUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.xiva.DataApiXivaUtils;
import ru.yandex.chemodan.app.dataapi.core.xiva.XivaUrlGeneratingTestUtils;
import ru.yandex.chemodan.app.dataapi.web.test.ApiTestBase;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

/**
 * @author metal
 */
@ActivateDataApiEmbeddedPg
public class XivaSubscriptionActionsTest extends ApiTestBase {
    private static final long UID = 101;

    @Value("${xiva.host}")
    private String xivaHost;
    @Value("${xiva.ya-team.host}")
    private String xivaYaTeam;

    @Override
    protected DataApiUserId createUser() {
        return new DataApiPassportUserId(UID);
    }

    @Override
    protected String getNamespace() {
        return ru.yandex.chemodan.util.web.NS.API;
    }

    @Test
    public void getStreamSubscriptionUrl() {
        checkStreamSubscriptionUrl(uid, xivaHost);
    }

    @Test
    public void getStreamSubscriptionUrlForYaTeam() {
        checkStreamSubscriptionUrl(new DataApiYaTeamUserId(1120000000000001L), xivaYaTeam);
    }

    private void checkStreamSubscriptionUrl(DataApiUserId uid, String xivaHost) {
        String url = "/databases/subscribe" +
                "?__uid=" + uid.toString() +
                "&oauth_token=oauthToken1" +
                "&client=client1" +
                "&session=session1" +
                "&database_ids=db_id1,db_id2";

        MockHttpServletResponse resp = sendRequestUsual("GET", url, "");

        String expectedUrl = xivaHost + "/v1/subscribe" +
                "?oauth_token=oauthToken1" +
                "&uid=" + DataApiXivaUtils.toXivaRecipient(uid) +
                "&client=client1" +
                "&session=session1" +
                "&service=datasync%3Adb_id1%2Bdb_id2";

        XivaUrlGeneratingTestUtils.assertUrlEquals(expectedUrl, getUrl(resp));
    }

    @Test
    public void getApplicationSubscriptionUrl() {
        String url = "/databases/subscribe/app" +
                "?__uid=" + uid.toString() +
                "&app=appOauthToken1" +
                "&session=session1" +
                "&platform=platform1" +
                "&platform_push_token=platformPushToken1" +
                "&database_ids=db_id1,db_id2";

        MockHttpServletResponse resp = sendRequestUsual("GET", url, "");

        String expectedUrl = xivaHost + "/v1/subscribe/app" +
                        "?oauth=appOauthToken1" +
                        "&session=session1" +
                        "&platform=platform1" +
                        "&platform_push_token=platformPushToken1" +
                        "&service=datasync%3Adb_id1%2Bdb_id2";

        XivaUrlGeneratingTestUtils.assertUrlEquals(expectedUrl, getUrl(resp));
    }

    @Test
    public void getCallbackSubscriptionUrl() {
        checkCallbackSubscriptionUrl(uid, xivaHost);
    }

    @Test
    public void getCallbackSubscriptionUrlForYaTeam() {
        checkCallbackSubscriptionUrl(new DataApiYaTeamUserId(1120000000000001L), xivaYaTeam);
    }

    private void checkCallbackSubscriptionUrl(DataApiUserId uid, String xivaHost) {
        String url = "/databases/subscribe/url" +
                "?__uid=" + uid.toString() +
                "&ctoken=xiva_ctoken" +
                "&callback=callback1" +
                "&session=session1" +
                "&database_ids=db_id1,db_id2";

        MockHttpServletResponse resp = sendRequestUsual("GET", url, "");

        String expectedUrl = xivaHost + "/v1/subscribe/url" +
                "?uid=" + DataApiXivaUtils.toXivaRecipient(uid) +
                "&ctoken=xiva_ctoken" +
                "&callback=callback1" +
                "&session=session1" +
                "&service=datasync%3Adb_id1%2Bdb_id2";

        XivaUrlGeneratingTestUtils.assertUrlEquals(expectedUrl, getUrl(resp));
    }

    private String getUrl(MockHttpServletResponse resp) {
        String response = getContent(resp);
        JsonNode node = JsonNodeUtils.getNode(response);
        return node.get("result").get("url").asText();
    }
}
