package ru.yandex.chemodan.app.dataapi.worker.dump;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.RetryPolicy;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.CollectionNamesExtractor;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.DataSyncDumper;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.FullDumpDatabaseUsersProcessor;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.YtCollectionNamesExtractor;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.YtDumpAndChangesMergeScriptManager;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.YtDumpAndChangesMerger;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.YtDumpBackupManager;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.YtDumpConsistencyManager;
import ru.yandex.chemodan.app.dataapi.worker.dump.full.YtDumpConsistencySupporterTask;
import ru.yandex.commune.bazinga.BazingaWorkerApp;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.inside.yt.kosher.Yt;

/**
 * @author metal
 */
@Configuration
public class DumpConfiguration {
    private final DynamicProperty<Integer> threadCount = DynamicProperty.cons("yt-dump-executor-thread-count", 8);

    @Bean
    public DumpDatabaseCronTasksRegistry dumpCronTasksRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager,
            DataApiManager dataApiManager,
            DiskDataSource diskDataSource,
            BazingaWorkerApp bazingaWorkerApp,
            Yt yt,
            @Qualifier("databaseOperationRetryPolicy")
            RetryPolicy dbRetryPolicy,
            @Qualifier("ytOperationRetryPolicy")
            RetryPolicy ytRetryPolicy,
            @Value("${dataapi.dump.database.max.user.number.in.yt.table.chunk}")
            int maxNumberOfUsersInSingleChunk,
            @Value("${dataapi.dump.database.max.skipped.users.count}")
            int maxSkippedUsers,
            @Value("${dataapi.dump.database.threads:-4}")
            int threads)
    {
        DumpDatabaseCronTasksRegistry dumpCronTasksRegistry = new DumpDatabaseCronTasksRegistry(
                zkRoot.child("datasync_dump_cron_tasks"),
                dataApiManager, diskDataSource, bazingaWorkerApp.getWorkerTaskRegistry(), yt,
                maxNumberOfUsersInSingleChunk, maxSkippedUsers, dbRetryPolicy, ytRetryPolicy, threads);
        zkManager.addClient(dumpCronTasksRegistry);
        return dumpCronTasksRegistry;
    }

    @Bean
    public DataSyncDumper dataSyncDumper(
            DataApiManager dataApiManager,
            DiskDataSource diskDataSource,
            Yt yt,
            @Value("${dataapi.dump.database.max.user.number.in.yt.table.chunk}")
            int maxNumberOfUsersInSingleChunk,
            @Value("${dataapi.dump.database.max.skipped.users.count}")
            int maxSkippedUsers,
            @Qualifier("databaseOperationRetryPolicy")
            RetryPolicy dbRetryPolicy,
            @Qualifier("ytOperationRetryPolicy")
            RetryPolicy ytRetryPolicy,
            AppSettingsRegistry appSettingsRegistry,
            @Value("${dataapi.dump.full.database.threads:-20}")
            int threads)
    {
        return new DataSyncDumper(
                dataApiManager,
                diskDataSource,
                yt,
                maxNumberOfUsersInSingleChunk, maxSkippedUsers, dbRetryPolicy, ytRetryPolicy, appSettingsRegistry, threads);
    }

    @Bean
    public CollectionNamesExtractor collectionNamesExtractor(Yt yt) {
        return new YtCollectionNamesExtractor(yt, FullDumpDatabaseUsersProcessor.DATASYNC_FULL_DUMP_YT_PATH_PREFIX);
    }

    @Bean
    public YtDumpBackupManager ytDumpBackupManager(
            Yt yt,
            @Qualifier("ytOperationRetryPolicy")
            RetryPolicy ytRetryPolicy,
            @Value("${dataapi.fulldump.number.of.days.to.backup}")
            int numberOfDaysToBackup)
    {
        return new YtDumpBackupManager(yt, ytRetryPolicy, numberOfDaysToBackup);
    }

    @Bean
    public YtDumpAndChangesMergeScriptManager ytDumpAndChangesMergeScriptManager(
            Yt yt,
            @Qualifier("ytOperationRetryPolicy")
            RetryPolicy ytRetryPolicy,
            @Value("${dataapi.fulldump.merge.script.local.path}")
            String mergeScriptLocalPath)
    {
        return new YtDumpAndChangesMergeScriptManager(yt, ytRetryPolicy, mergeScriptLocalPath);
    }

    @Bean
    public YtDumpAndChangesMerger ytDumpAndChangesMerger(
            Yt yt,
            @Qualifier("ytOperationRetryPolicy")
            RetryPolicy ytRetryPolicy)
    {
        return new YtDumpAndChangesMerger(yt, ytRetryPolicy);
    }

    @Bean
    public YtDumpConsistencyManager ytDumpConsistencyManager(YtDumpAndChangesMerger ytDumpAndChangesMerger,
            YtDumpAndChangesMergeScriptManager ytDumpAndChangesMergeScriptManager,
            CollectionNamesExtractor collectionNamesExtractor, YtDumpBackupManager ytDumpBackupManager,
            AppSettingsRegistry appSettingsRegistry,
            @Qualifier("dumpExecutorService") ExecutorService executorService)
    {
        return new YtDumpConsistencyManager(ytDumpAndChangesMerger, ytDumpAndChangesMergeScriptManager,
                collectionNamesExtractor, ytDumpBackupManager,
                appSettingsRegistry, executorService);
    }

    @Bean
    public YtDumpConsistencySupporterTask ytDumpConsistencySupporterTask(
            YtDumpConsistencyManager ytDumpConsistencyManager)
    {
        return new YtDumpConsistencySupporterTask(ytDumpConsistencyManager);
    }


    @Bean(name = "dumpExecutorService")
    public ExecutorService dumpExecutorService(DynamicPropertyManager dynamicPropertyManager) {
        ThreadPoolExecutor executor = new ThreadPoolExecutor(threadCount.get(), threadCount.get(),
                1, TimeUnit.MINUTES, new LinkedBlockingQueue<>());

        dynamicPropertyManager.registerWatcher(threadCount, value -> {
            executor.setMaximumPoolSize(value);
            executor.setCorePoolSize(value);
        });
        return executor;
    }

}
