package ru.yandex.chemodan.app.dataapi.worker.dump.full;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import net.jodah.failsafe.RetryPolicy;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppDatabaseSettings;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.worker.dump.AbstractDumpDatabaseUsersProcessor;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


/**
 * @author metal
 * @author vpronto
 */
public class DataSyncDumper implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(DataSyncDumper.class);

    private final DataApiManager dataApiManager;
    private final DiskDataSource diskDataSource;

    private final Yt yt;
    private final int maxNumberOfUsersInSingleChunk;

    private final int maxSkippedUsers;
    private final int threads;

    private final RetryPolicy dbRetryPolicy;
    private final RetryPolicy ytRetryPolicy;

    private final AppSettingsRegistry appSettingsRegistry;
    private final ExecutorService executorService = Executors.newSingleThreadExecutor();

    public DataSyncDumper(
            DataApiManager dataApiManager,
            DiskDataSource diskDataSource,
            Yt yt, int maxNumberOfUsersInSingleChunk, int maxSkippedUsers,
            RetryPolicy dbRetryPolicy, RetryPolicy ytRetryPolicy,
            AppSettingsRegistry appSettingsRegistry,
            int threads)
    {
        this.dataApiManager = dataApiManager;
        this.diskDataSource = diskDataSource;
        this.yt = yt;
        this.maxNumberOfUsersInSingleChunk = maxNumberOfUsersInSingleChunk;
        this.maxSkippedUsers = maxSkippedUsers;
        this.dbRetryPolicy = dbRetryPolicy;
        this.ytRetryPolicy = ytRetryPolicy;
        this.appSettingsRegistry = appSettingsRegistry;
        this.threads = threads;
    }

    public void dump() {
        getDatabasesToDump().forEach(this::dumpSpecificDatabase);
    }

    public void dump(String app, ListF<String> dbs) {
        getDatabasesToDump(app, dbs).forEach(this::dumpSpecificDatabase);

    }

    public void dumpSpecificDatabase(DatabaseRef databaseRef) {
        dumpSpecificDatabase(databaseRef.appNameO(), databaseRef.databaseId(), Option.empty());
    }

    public void dumpSpecificDatabase(Option<String> appName, String databaseId, Option<String> collection) {
        dumpSpecificDatabase(appName, databaseId, collection, Option.empty(), Option.empty());
    }

    public void dumpSpecificDatabase(Option<String> appName, String databaseId, Option<String> collection,
            Option<Integer> threadsOverride, Option<Integer> maxNumberOfUsersInSingleChunkOverride) {
        executorService.submit(() -> {
            AbstractDumpDatabaseUsersProcessor dumpDatabaseUsersProcessor = new FullDumpDatabaseUsersProcessor(
                    dataApiManager, diskDataSource,
                    appName, databaseId, collection,
                    yt, maxNumberOfUsersInSingleChunkOverride.getOrElse(maxNumberOfUsersInSingleChunk),
                    maxSkippedUsers, dbRetryPolicy, ytRetryPolicy, threadsOverride.getOrElse(threads));
            dumpDatabaseUsersProcessor.process();

            logger.info("Number of skipped users for database {}, {} is {}", databaseId, appName,
                    dumpDatabaseUsersProcessor.getSkippedUsers());
        });
    }

    private Iterable<DatabaseRef> getDatabasesToDump(String app, ListF<String> dbs) {
        return appSettingsRegistry
                .getAll()
                .filter(a -> a.appName.isSome(app) && a.databaseId.isMatch(s -> dbs.containsF().apply(s)))
                .map(settings -> DatabaseRef.cons(settings.appName, settings.databaseId.get()));
    }

    private ListF<DatabaseRef> getDatabasesToDump() {
        return appSettingsRegistry
                .getAll()
                .filter(AppDatabaseSettings::isHaveDumpInYt)
                .map(settings -> DatabaseRef.cons(settings.appName, settings.databaseId.get()));
    }

    @Override
    public void close() throws Exception {
        executorService.shutdown();
    }
}
